# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class Balise
  #Point along track that executes custom ruby code when train goes over it.
  
  # Regular expression to check if a line of code is considered safe.
  # TODO: allow timer. Allow condition (train name only?)  at end of line.
  SAFE_LINE = %r{
    \A
      (\#.*) |               # Comment.
      (                      # Variable assigner.
         \w*\s?=\s?
         (
           "\w*" |
           '\w*' |
           Sketchup\.active_model\.materials |  # List all materials in model.
           \w*\[rand\(\w*\.length-1\)\] |       # Pick random from array.
           \w*\.sample |
           balise\.track
         )
      ) |
      (                      # Run method on variable (e.g. 'train').
        (UI\.start_timer\(\d*\.?\d*,\s?false\)\s?\{)?
        (\w+)\.
        (
          ((v|v_target|a)\s?\*?=\s?\d*\.?\d*) |
            reverse |
            r_stocks\.?
            (first|\[\d*\]|(each\s?\{\s?\|\s?\w*\s?\|\s*\w*))\.
            (
              group\.material\s?=\s?(\"?\w*\"?) |
              group.erase!
            )?\s?(end|\})? |
            set_switch\(
            (
              \d |
              train\.name\s?==[a-zA-Z0-9_\s\"\']*\?\s?\d\s?:\s?\d
            )
          \)
        )?
        \s?\}?
      )
    (\s*if\s?\(?train\.name\s?==\s?[a-zA-Z0-9_\s\"\']+\)?)?
    (\z|\#)
  }x
  
  #All loaded rolling stock objects. In mac these might be in different models
  @@instances ||= []

  #Class variable accessors

  def self.instances; @@instances; end
  def self.instances=(value); @@instances = value; end
  
  #Class methods

  def self.load_from_file(model)
    #Create balise objects from attribute dictionary
    #Find track object from point since custom objects cannot be stored in attributes
        
    a = model.get_attribute ID, "balise_list"
    new_balises = []
    if a
      a.each do |child_a|
        #Manually list all objects that should be retrieved by correct index. cannot use hash in attribute
        point = child_a[0]
        code = child_a[1]
        #Check that the same balise doesn't already exist which may be the case when reloading main.rb with model open
        #If balise has been repositioned since last save the old one will be added anyway
        unless @@instances.find { |b| b.point == point }
          new_balises << Balise.new(point, nil, code)
        end
      end
    end
    
    # Perform security check on loaded balises.
    security_check new_balises, :model
    
    nil
    
  end
  
  def self.save_to_file(model)
    #Save all balise objects to attribute dictionary in model
    #Track cannot be retrieved though since it's lost when saving to attribute dictionary
    
    a = []
    @@instances.each do |b|
      #Manually list all objects that should be saved in correct order. cannot use hash in attribute
      child_a = [b.point, b.code]
      a << child_a
    end
    
    model.set_attribute ID, "balise_list", a
    
  end
  
  # Public: Check if a string is safe (false) or potentially harmful (true) by
  # comparing to regular expression of known safe code strings.
  #
  # code - The string to test.
  #
  # Returns boolean, true means potentially harmful, false safe.
  def self.potentially_harmful?(code)
  
    # TODO: UNDER CONSTRUCTION: remove comments before checking. check lines individually. add more safe ode examples to regex, false positives make users stop reading these messages!! also add license/warranty info somewhere.
    code.split("\n").any? { |l| l !~ SAFE_LINE }
    
  end
  
  # Public: Check array of balises for potentially harmful ones.
  # If any such are found prompt user what to do with them.
  # Either keep or remove them.
  # Runs when loading model and placing track formations.
  #
  # balises - Array of balises to check.
  # context - Where the check is performed, affects text shown to user.
  #           Either :model, :track_formation or nil (default: nil).
  #
  # Returns nothing.
  def self.security_check(balises, context)
  
    unsafe = balises.select { |b| potentially_harmful? b.code }
    unless unsafe.empty?
      title = "#{S.tr("Eneroth Railroad System")} - #{S.tr("Security Notification")}"
      dlg = UI::WebDialog.new(title, false, "#{ID}_balise_warning", 550, 350, 500, 100, true)
      dlg.min_width = 550
      dlg.min_height = 350
      dlg.navigation_buttons_enabled = false
      dlg.set_background_color dlg.get_default_dialog_color
      dlg.set_file(File.join(DIALOG_DIR, "balise_warning", "index.html"))
      
      msg = "This #{context == :track_formation ? "track formation" : "model"} contains potentially harmful executable code (triggered when trains passes certain point(s)). Do you want to disable this code? Only allow from trusted sources."
      js = "document.getElementById('notification').innerHTML = '#{msg}';"
      code = ""
      unsafe.each do |b|
        b_code = b.code.gsub("'", "&#39;").gsub("<", "&lt;").gsub(">", "&gt;").gsub("\r\n", "\n").gsub("\n", "<br />")
        coords = "(#{b.point.x.to_l};#{b.point.y.to_l};#{b.point.z.to_l})"
        code << "<div>#{coords} <code>#{b_code}</code></div>"
      end
      js << "document.getElementById('code').innerHTML = '#{code}';"
      
      allow_when_close = false
      dlg.add_action_callback("close") do
        dlg.close
      end
      dlg.add_action_callback("allow") do
        allow_when_close = true
        dlg.close
      end
      dlg.set_on_close do
        unsafe.each { |b| b.delete! } unless allow_when_close
      end

      dlg.show_modal { dlg.execute_script js }
    end
    
    nil
    
  end
  
  #Instance attribute accessors
  
  attr_accessor :point
  attr_accessor :code
  attr_accessor :track
  
  def initialize(point, track = nil, code = nil)
  
    track ||= Track.inspect_point(point, Sketchup.active_model)[:track]
  
    @point = point
    @code = code || ""
    @track = track
    
    @last_run = nil
    @last_train = nil
    
    @@instances << self
    
  end

  def code_editor
    #Create a web dialog that lets user write the code
  
    #input = UI.inputbox ["Code"], [@code], "Edit Custom Code"
    #return unless input
    #@code = input[0]
    
    #Create web dialog
    dialog = UI::WebDialog.new(S.tr("Code Editor"), false, "#{ID}_balise_code_editor", 640, 400, 300, 100, false)
    dialog.navigation_buttons_enabled = false
    dialog.set_background_color dialog.get_default_dialog_color
    dialog.set_file(File.join(DIALOG_DIR, "balise", "index.html"))
    
    #Translate strings
    js = S.tr_dlg
    
    #Add current data to form
    code = @code.gsub("\n", "\\n\\r").gsub("\r", "").gsub("\"", "\\\"")#escape special characters
    js << "document.getElementById('code_editor').value=\"#{code}\";"
    js << "document.getElementById('code_editor').focus();"

    #Show dialog
    if WIN
      dialog.show { dialog.execute_script js }
    else
      dialog.show_modal { dialog.execute_script js }
    end
    
    #Apply changes
    dialog.add_action_callback("apply") { |dialog, callbacks|
    
      @code = dialog.get_element_value("code_editor")
      
      #Close if user pressed OK rather than Apply
      dialog.close if callbacks == "close"
    }
    
    #Close web dialog
    dialog.add_action_callback("close") {
      dialog.close
    }

    #Show help file
    dialog.add_action_callback("show_help") {
      UI.openURL("file://" + File.join(DOCS_DIR, "tool_balise.html"))
    }
    
    #Open link in default browser
    dialog.add_action_callback("open_in_default_browser") { |dialog, callbacks|
      UI.openURL callbacks
    }
    
  end
  
  def delete!
  
    @@instances.delete self
  
  end
  
  def execute(train)
    #Execute the code stored as a string.
    #Called from Train.go when train moves over point
    
    #'train' is the given reference to the train object
    
    #'balise' is the reference to the balise object.
    #User could also write 'self' since code is executed within this namespace but balise makes more sense.
    balise = self
    
    #'last_train' and 'last_run' are the references to the previous train passing the balise and the time when it happened.
    #User could leave out leading '@' but this is more user intuitive.
    last_train = @last_train
    last_run = @last_run
    
    eval @code if @code
    
    @last_train = train
    @last_run = Time.now
    
  end
  
  def inspect
    #Do not show attributes when inspecting. It clutters the console
    self.to_s
  end
  
end#class

end#module
