# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolBalise
  #This tool is used to create and edit balies.
  #A balise is an object along the track that executes some custom code when train passes.
  
  #Sketchup tool definitions
    
  def initialize
  
    #Balise object currently being moved
    @being_moved
    
    #Balise currently hovered (set in mousemove)
    @being_hovered
    
    #Statusbar texts
    @status_text = S.tr "Click track to add. Click existing to move, then press delete to remove. Double click to edit."
    
    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "balise.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def activate
  
    #Check so there are tracks in model, otherwise exit tool
    unless Track.instances.any? { |t| t.model == Sketchup.active_model }
      UI.messagebox(S.tr("There are no tracks in this model.") + "\n" + S.tr("One or more tracks must be added before adding balises."))
      Sketchup.send_action "selectSelectionTool:"
    end

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    Sketchup.set_status_text(@status_text, SB_PROMPT)

  end
  
  def deactivate(view)
  
    view.invalidate

  end
  
  def draw(view)
  
    #Draw all balises but the hovered one and one being moved
    color = Sketchup::Color.new "Yellow"
    color_border = Sketchup::Color.new [127,127,0]
    
    all_balises_but_selected_and_hovered = Balise.instances.select { |b| ![@being_moved, @being_hovered].include?(b) }
    
    unless all_balises_but_selected_and_hovered.empty?
      view.draw_points(all_balises_but_selected_and_hovered.map { |b| b.point } , 10, 2, color)
      view.draw_points(all_balises_but_selected_and_hovered.map { |b| b.point } , 10, 1, color_border)
    end
    
    #Draw hovered and moved one
    color = Sketchup::Color.new "Blue"
    color_border = Sketchup::Color.new "DarkBlue"
    
    if @being_moved
      #Draw balise being moved if any
      
      view.draw_points([@being_moved.point], 10, 2, color)
      view.draw_points([@being_moved.point], 10, 1, color_border)
      
    elsif @being_hovered
      #Draw balise being hovered if any
    
      view.draw_points([@being_hovered.point], 10, 2, color)
      view.draw_points([@being_hovered.point], 10, 1, color_border)
      view.tooltip = @being_hovered.code + "\n\n" + S.tr("Double click to edit.")
      
    else
      #Draw the point where a new balise would be added if user clicks
      
      picked = @ph.best_picked
      if Track.group_is_track? picked
        #A track is hovered
        point = Track.inspect_point(@ip.position)[:point]
        color = Sketchup::Color.new "Yellow"
        view.draw_points([point], 10, 1, color)
      end
    end
    
  end

  def onMouseMove(flags, x, y, view)
  
    @ip.pick view, x, y
    @ph.do_pick(x, y)

    if @being_moved
      #Moving balise
      
      q = Track.inspect_point(@ip.position)
      @being_moved.track = q[:track]
      @being_moved.point = q[:point]
    
    else
      #Find hovered balise if any
      
      hovered_array = Balise.instances.select { |b| EneRailroad.mouse_on_point?(view, x, y, b.point) }
      @being_hovered = hovered_array.empty? ? nil : hovered_array[0]
    end
    
    view.invalidate
  
  end
  
  def onLButtonDown(flags, x, y, view)
  
    #Save position for mouse down so it can be compared to mouse up to see if mouse was dragged
    @x_down = x
    @y_down = y
  
    if @being_moved
      #A balise stops being moved
      
      @being_moved = nil
    
    elsif @being_hovered
      #An existing balise was clicked, move it
      
      @being_moved = @being_hovered
      @being_hovered = nil
      
      #Clear selection so pressing delete wont delete any entity, just the balise moved
      Sketchup.active_model.selection.clear
      
    else
      #Find point along track and create new balise
      
      @ph.do_pick(x, y)
      picked = @ph.best_picked
      
      if picked && Track.group_is_track?(picked)
        #A track was clicked
            
        q = Track.inspect_point(@ip.position)
        track = q[:track]
        point = q[:point]
        
        b = Balise.new point, track
        b.code_editor
      
      end
      
    end
    
  end
  
  def onLButtonUp(flags, x, y, view)

    if @being_moved && (@x_down - x).abs > 10 && (@y_down - y).abs > 10
      #Mouse has to be moved 10px in x or y for it to count as dragged
      #Stop moving balise, same code as on mouse down
      
      @being_moved = nil
      
    end
    
  end
  
  def onLButtonDoubleClick(flags, x, y, view)
  
    #Take reference to clicked balise from @being_moved since mousedown has already been called
    #Double clicking when a balie is already being moved causes being_moved to be nil since mousedown stops it from being moved.
    b = @being_moved
    if b
      @being_moved = nil
      b.code_editor
    end
  
  end
  
  def onKeyUp(key, repeat, flags, view)
    
    if key == VK_DELETE 
      @being_moved.delete!
      @being_moved = nil
      
      view.invalidate
    end
    
  end
  
  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate
  end
  
end#class

end#module
