# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

# Public: Name of attribute dictionary used for 3d text.
ATTR_DICT_TEXT = "#{ID}_text"

# Internal: path to cursor directory.
CURSOR_DIR = File.join PLUGIN_DIR, "cursors"

# Internal: path to documentation directory.
DOCS_DIR = File.join PLUGIN_DIR, "docs"

# Internal: Path to directory for web dialogs.
DIALOG_DIR = File.join PLUGIN_DIR, "dialogs"

# Internal: Horizontal plane with origin in.
PLANE_HORIZONTAL = [ORIGIN, Z_AXIS]

# Internal: Whether SU is currently running in windows.
WIN = RUBY_PLATFORM !~ /darwin/

# Include all classes and modules.
###require File.join(PLUGIN_DIR, "lang.rb")# Language handler already loaded.
require File.join(PLUGIN_DIR, "win32api.rb")
require File.join(PLUGIN_DIR, "my_geom.rb")
require File.join(PLUGIN_DIR, "my_view.rb")
require File.join(PLUGIN_DIR, "upright_extruder.rb")
require File.join(PLUGIN_DIR, "template.rb")
require File.join(PLUGIN_DIR, "track.rb")
require File.join(PLUGIN_DIR, "track_insert_tool.rb")
require File.join(PLUGIN_DIR, "track_position_tool.rb")
require File.join(PLUGIN_DIR, "track_switch_tool.rb")
require File.join(PLUGIN_DIR, "structure.rb")
require File.join(PLUGIN_DIR, "structure_insert_tool.rb")
require File.join(PLUGIN_DIR, "structure_position_tool.rb")
require File.join(PLUGIN_DIR, "r_stock.rb")
require File.join(PLUGIN_DIR, "r_stock_insert_tool.rb")
require File.join(PLUGIN_DIR, "train.rb")
require File.join(PLUGIN_DIR, "train_coupling_tool.rb")
require File.join(PLUGIN_DIR, "train_drive_tool.rb")
require File.join(PLUGIN_DIR, "animate.rb")
require File.join(PLUGIN_DIR, "balise.rb")
require File.join(PLUGIN_DIR, "balise_tool.rb")
require File.join(PLUGIN_DIR, "observers.rb")

# Some general methods that don't really fit into any class or module.

# internal: Set language for UI.
#
# lang - String language code or nil when language should be guessed.
#
# Returns noting.
def self.change_lang(lang)
  #Change what language to use in UI.
  #When lang is false or nil, guess language.
  
  return if lang == S.l10n
  
  Sketchup.write_default(ID, "language", lang)
  lang = S.guess_l10n unless lang
  S.load lang

  UI.messagebox(S.tr("Sketchup needs to be restarted to fully use new language."))
  
  nil
  
end

# Internal: Open a directory in file browser.
#
# path - Path to directory.
#
# Returns nothing.
def self.open_dir(path)

  if WIN
    path = path.gsub "/", "\\"
    system("explorer.exe \"#{path}\"")
  else
    system("open \"#{path}\"")
  end

  nil

end

# Internal: Edit 1-dimensional hash in dialog box.
# 
# hash - The hash to edit.
#
# Returns edited Hash if user clicks OK or nil if user cancels.
def self.prompt_hash_values(hash)
  
  # Separate keys and values.
  keys = hash.keys
  values = hash.values
  
  # Prompt for new values.
  keys_s = keys.map { |i| i.to_s }# to_s since keys may be symbols.
  values_s = values.map { |i| i.to_s }
  title = S.tr "Edit Values"
  input = UI.inputbox(keys_s, values_s, title)
  return unless input
  
  # Merge into hash.
  Hash[*keys.zip(input).flatten]
  
end

# Internal: Check if given 2d coordinates are on a 3d point in given view.
#
# view  - The view to check point in.
# x     - 2d X coordinate.
# Y     - 2d Y coordinate.
# point - The 3d point.
#
# Returns true when points matches, otherwise false.
def self.mouse_on_point?(view, x, y, point)

  s_point = view.screen_coords point
  
  (s_point.x - x).abs < 10 && (s_point.y - y).abs < 10

end

# IO

def self.file_2_object path
  #Load object from file where it's been saved as marshal string

  return unless File.exists? path
  ###string = IO.read(info_file) fails when reaching special characters
  string = open(path, "rb") { |io| io.read }
  return unless string
  
  Marshal::load(string)
  
end

def self.object_2_file(path, object)
    #Save object to file as marshal string
    
    File.open(path, "w+") { |f| Marshal.dump(object, f) }
    
end

def self.object_2_json(object)
  #Turn object into JSON string

  case object
  when Array
    s = "["
    s << object.map { |o| object_2_json(o) }.join(", ")
    s << "]"
  when Hash
    s = "{"
    a = []
    object.each_pair { |k, v| a << "\"#{k.to_s}\": #{object_2_json(v)}" }
    s << a.join(", ")
    s << "}"
  when Geom::Point3d, Geom::Vector3d
    s = object.to_a.inspect
  when NilClass
    s = "null"
  else
    s = object.inspect
  end
  
  s
  
end

# Drawing to model.

def self.write_text(group, text, font = nil)
  #Draw 3d text to model
  
  #Text is drawn to a pre-drawn group and is 1 m high in the groups internal coordinate system.
  #Text is the x-y plane (red, green) of the group.
  #First manually draw a group with a 1 m long edge (or other entity) along y axis (green), then position and scale the group as you want the text to be.
  #Text is centered on group origin.
  #Remember to locate the group a small distance (e.g. 1 mm) outside the face the text should be drawn 'on' to avoid z-fighting.
  
  font ||= "arial"
  
  #Empty group
  group.entities.clear!
  
  #Add a new group to add text to, this group will later be centered
  text_group = group.entities.add_group
  
  #Draw text
  text_group.entities.add_3d_text(text, TextAlignCenter, font, true, false, 1.m, 0.0, 0, true, 0)
  
  #Hide text edges
  text_group.entities.each do |e|
    next unless e.class == Sketchup::Edge
    e.hidden = true
  end#each
  
  #Center text group in parent group
  bb = text_group.local_bounds
  p_center = bb.center
  vector = Geom::Point3d.new - p_center
  trans = Geom::Transformation.translation vector
  text_group.transform! trans
      
end

# Dev stuff.
# Adding or updating track types, rolling stocks, signals etc.

# Internal: Reload whole extension (except loader) without littering console.
# Inspired by ThomTohm's method.
#
# Returns nothing.
def self.reload

  # Hide warnings for already defined constants.
  old_verbose = $VERBOSE
  $VERBOSE = nil

  # Load
  Dir.glob(File.join(PLUGIN_DIR, "*.rb")).each { |f| load f }

  $VERBOSE = old_verbose

  nil

end

# Menus and toolbars.
file = __FILE__
unless file_loaded? file

  #Context menu
  UI.add_context_menu_handler do |menu|
    ani = Animate.get_from_model
    if t = Track.get_from_selection
      #Track is selected
      
      menu.add_separator
      menu.add_item(S.tr("Track Position")) { Sketchup.active_model.select_tool ToolTrackPosition.new(t) }
      item = menu.add_item(S.tr("Set Switch State")) { Sketchup.active_model.select_tool ToolTrackSwitch.new(t) }
      menu.set_validation_proc(item) { t.is_switch? ? MF_ENABLED : MF_GRAYED }
      menu.add_item(S.tr("Track Properties")) { Track.properties_dialog t }
      
    elsif str = Structure.get_from_selection
      #Structure is selected
      
      menu.add_separator
      menu.add_item(S.tr("Structure Position")) { Sketchup.active_model.select_tool ToolStructurePosition.new(str) }
      menu.add_item(S.tr("Structure Properties")) { Structure.properties_dialog str }
    
    elsif rs = RStock.get_from_selection
      #Rolling stock is selected
      
      menu.add_separator
      menu.add_item(S.tr("Drive Train")) { Sketchup.active_model.select_tool ToolTrainDrive.new(rs.train); Sketchup.active_model.selection.clear }
      item = menu.add_item(S.tr("Follow Rolling Stock With Camera")) { ani.followed = (ani.followed == rs ? nil : rs) }
      menu.set_validation_proc(item) { ani.followed == rs ? MF_CHECKED : MF_ENABLED }
      submenu = menu.add_submenu(S.tr("Change Texts"))
      submenu.add_item(S.tr("This Rolling Stock")) { rs.text_UI }
      submenu.add_item(S.tr("Whole Train")) { rs.train.text_UI }
      menu.add_item(S.tr("Save Rolling Stock To Library")) { Template.save_selected_r_stocks }
      
    end
    
    has_seperator = false
    
    #Stop following rolling stock (only if one is followed)
    #NOTE: SU ISSUE: not visible when clicking outside entities -_-
    if ani.followed && ani.followed != RStock.get_from_selection
      menu.add_separator
      has_seperator = true
      menu.add_item(S.tr("Stop Following Rolling Stock with Camera")) { ani.followed = nil }
    end
    
    #Properties for multiple tracks
    ss_tracks = Sketchup.active_model.selection.map { |e| EneRailroad::Track.get_from_group e }
    if ss_tracks.all?
      ss_tracks.compact!
      if ss_tracks.length > 1
        menu.add_separator unless has_seperator      
        menu.add_item(S.tr("Track Properties")) { Track.properties_dialog ss_tracks }
      end
    end
    
    #Properties for multiple structures
    ss_structures = Sketchup.active_model.selection.map { |e| EneRailroad::Structure.get_from_group e }
    if ss_structures.all?
      ss_structures.compact!
      if ss_structures.length > 1
        menu.add_separator unless has_seperator      
        menu.add_item(S.tr("Structure Properties")) { Structure.properties_dialog ss_structures }
      end
    end
    
    #Save multiple rolling stocks to library
    ss_rss = Sketchup.active_model.selection.map { |e| EneRailroad::RStock.get_from_group e }
    if ss_rss.all?
      ss_rss.compact!
      if ss_rss.length > 1
        menu.add_separator unless has_seperator
        menu.add_item(S.tr("Save Rolling Stocks To Library")) { Template.save_selected_r_stocks }
      end
    end
    
  end

  #Menu bar
  menu = UI.menu("Plugins").add_submenu(S.tr(JPOD ? "JPods" : "Eneroth Railroad System"))

  menu.add_item(S.tr("Add Track")) { Sketchup.active_model.select_tool ToolTrackInsert.new }
  menu.add_item(S.tr("Track Position")) { Sketchup.active_model.select_tool ToolTrackPosition.new }
  menu.add_item(S.tr("Set Switch State")) { Sketchup.active_model.select_tool ToolTrackSwitch.new }
  item = menu.add_item(S.tr("Track Properties")) { Track.properties_dialog(Sketchup.active_model.selection.map { |e| Track.get_from_group e }) }
  menu.set_validation_proc(item) { ss=Sketchup.active_model.selection; !ss.empty? && ss.all? { |e| Track.get_from_group e } ? MF_ENABLED : MF_GRAYED }
  
  menu.add_item(S.tr("Add Structure")) { Sketchup.active_model.select_tool ToolStructureInsert.new }
  menu.add_item(S.tr("Structure Position")) { Sketchup.active_model.select_tool ToolStructurePosition.new }
  item = menu.add_item(S.tr("Structure Properties")) { Structure.properties_dialog(Sketchup.active_model.selection.map { |e| Structure.get_from_group e }) }
  menu.set_validation_proc(item) { ss=Sketchup.active_model.selection; !ss.empty? && ss.all? { |e| Structure.get_from_group e } ? MF_ENABLED : MF_GRAYED }
  
  menu.add_separator
  
  menu.add_item(S.tr("Add Rolling Stock")) { Sketchup.active_model.select_tool ToolRStockInsert.new }
  menu.add_item(S.tr("Couplings")) { Sketchup.active_model.select_tool ToolTrainCoupling.new }
  menu.add_item(S.tr("Drive Train")) { Sketchup.active_model.select_tool ToolTrainDrive.new }
  item = menu.add_item(S.tr("Save Rolling Stocks To Library")) { Template.save_selected_r_stocks }
  menu.set_validation_proc(item) { ss=Sketchup.active_model.selection; !ss.empty? && ss.all? { |e| RStock.get_from_group e } ? MF_ENABLED : MF_GRAYED }
  
  menu.add_separator
  
  item = menu.add_item(S.tr("Play")) { ani = Animate.get_from_model; ani.go = !ani.go }
  menu.set_validation_proc(item) { Animate.get_from_model.go ? MF_CHECKED : MF_ENABLED }
  
  menu.add_separator
  
  menu.add_item(S.tr("Balise")) { Sketchup.active_model.select_tool ToolBalise.new }
  
  menu.add_separator
  
  menu.add_item(S.tr("Advanced Animation Settings")) {  Animate.get_from_model.settings_dialog }
  
  menu.add_item(S.tr("Structure Part Info")) { Structure.part_info_dialog }
  
  if S.translations.length > 0
    l_menu = menu.add_submenu(S.tr("Change Language"))
    item = l_menu.add_item("English") { change_lang "en" }
    l_menu.set_validation_proc(item) { S.l10n == "en" ? MF_CHECKED : MF_ENABLED }
    S.translations.each_pair { |k, v|
      item = l_menu.add_item("#{v["name"]} (#{S.tr("By")} #{v["author"]})") { change_lang k }
      l_menu.set_validation_proc(item) { S.l10n == k ? MF_CHECKED : MF_ENABLED }
    }
  end
  
  menu.add_item(S.tr("Documentation")) { UI.openURL("file://" + File.join(DOCS_DIR, "home.html")) }

  #Toolbar
  tb = UI::Toolbar.new(S.tr(JPOD ? "JPods" : "Eneroth Railroad System"))

  cmd = UI::Command.new(S.tr("Add Track")) { Sketchup.active_model.select_tool ToolTrackInsert.new }
  cmd.large_icon = "toolbar_icons/track_insert.png"
  cmd.small_icon = "toolbar_icons/track_insert_small.png"
  cmd.tooltip = S.tr "Add Track"
  cmd.status_bar_text = S.tr "Insert a new track to model, copy existing tracks, offset tracks or connect loose track endings."
  tb.add_item cmd

  cmd = UI::Command.new(S.tr("Track Position")) { Sketchup.active_model.select_tool ToolTrackPosition.new }
  cmd.large_icon = "toolbar_icons/track_position.png"
  cmd.small_icon = "toolbar_icons/track_position_small.png"
  cmd.tooltip = S.tr "Track Position"
  cmd.status_bar_text = S.tr "Change track's position by moving control points."
  tb.add_item cmd

  cmd = UI::Command.new(S.tr("Set Switch State")) { Sketchup.active_model.select_tool ToolTrackSwitch.new }
  cmd.large_icon = "toolbar_icons/track_switch.png"
  cmd.small_icon = "toolbar_icons/track_switch_small.png"
  cmd.tooltip = S.tr "Set Switch State"
  cmd.status_bar_text = S.tr "Change what direction railroad switches are set to."
  tb.add_item cmd
  
  cmd = UI::Command.new(S.tr("Add Structure")) { Sketchup.active_model.select_tool ToolStructureInsert.new }
  cmd.large_icon = "toolbar_icons/structure_insert.png"
  cmd.small_icon = "toolbar_icons/structure_insert_small.png"
  cmd.tooltip = S.tr "Add Structure"
  cmd.status_bar_text = S.tr "Add structure along track. E.g. a station, tunnel, bridge or catenary."
  tb.add_item cmd

  cmd = UI::Command.new(S.tr("Structure Position")) { Sketchup.active_model.select_tool ToolStructurePosition.new }
  cmd.large_icon = "toolbar_icons/structure_position.png"
  cmd.small_icon = "toolbar_icons/structure_position_small.png"
  cmd.tooltip = S.tr "Structure Position"
  cmd.status_bar_text = S.tr "Change structure's position by moving control points or set length."
  tb.add_item cmd
  
  tb.add_separator

  cmd = UI::Command.new(S.tr("Add Rolling Stock")) { Sketchup.active_model.select_tool ToolRStockInsert.new }
  cmd.large_icon = "toolbar_icons/r_stock_insert.png"
  cmd.small_icon = "toolbar_icons/r_stock_insert_small.png"
  cmd.tooltip = S.tr "Add Rolling Stock"
  cmd.status_bar_text = S.tr "Insert a new rolling stock from a library or create a new from group."
  tb.add_item cmd

  cmd = UI::Command.new(S.tr("Couplings")) { Sketchup.active_model.select_tool ToolTrainCoupling.new }
  cmd.large_icon = "toolbar_icons/r_stock_couple.png"
  cmd.small_icon = "toolbar_icons/r_stock_couple_small.png"
  cmd.tooltip = S.tr "Couplings"
  cmd.status_bar_text = S.tr "Connect or disconnect rolling stocks."
  tb.add_item cmd

  cmd = UI::Command.new(S.tr("Drive Train")) { Sketchup.active_model.select_tool ToolTrainDrive.new }
  cmd.large_icon = "toolbar_icons/train.png"
  cmd.small_icon = "toolbar_icons/train_small.png"
  cmd.tooltip = S.tr "Drive Train"
  cmd.status_bar_text = S.tr "Drive train by controlling it's speed."
  tb.add_item cmd

  tb.add_separator

  cmd = UI::Command.new(S.tr("Run")) { ani = Animate.get_from_model; ani.go = !ani.go }
  cmd.large_icon = "toolbar_icons/animate.png"
  cmd.small_icon = "toolbar_icons/animate_small.png"
  cmd.tooltip = S.tr "Play/Pause"
  cmd.status_bar_text = S.tr "Start or stop animations."
  cmd.set_validation_proc { ani = Animate.get_from_model; ani && ani.go ? MF_CHECKED : MF_ENABLED }
  tb.add_item cmd

  tb.add_separator
  
  cmd = UI::Command.new(S.tr("Balise")) { Sketchup.active_model.select_tool ToolBalise.new }
  cmd.large_icon = "toolbar_icons/balise.png"
  cmd.small_icon = "toolbar_icons/balise_small.png"
  cmd.tooltip = S.tr "Balise"
  cmd.status_bar_text = S.tr "Add custom code that executes when train passes certain point."
  tb.add_item cmd

  UI.start_timer(0.1, false) { tb.restore }#Use timer as workaround for bug 2902434.

  file_loaded file
end

end# Module
