# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

# Internal: Various geometrical methods.
module MyGeom

  def self.angle_in_plane(v1, v2, plane = nil)
    #Get angle between vectors in a given plane.
    #Can return negative angles unlike built in angle_between method.
    #Return angle in radians in range from negative pi to positive pie including positive pie.
    #Use x-y plane if no plane is set.

    plane = PLANE_HORIZONTAL unless plane

    #Flatten to plane
    v1 = MyGeom.flatten_vector(v1, plane)
    v2 = MyGeom.flatten_vector(v2, plane)

    #Get angle between
    a = v1.angle_between v2

    #Determine if angle is positive or negative
    negAnglePointForward = Geom::Point3d.new.offset v2
    negAnglePointPos = negAnglePointForward.transform(Geom::Transformation.rotation(Geom::Point3d.new, plane[1], Math::PI/2))
    negAnglePointNeg = negAnglePointForward.transform(Geom::Transformation.rotation(Geom::Point3d.new, plane[1], -Math::PI/2))
    negAnglePointCheck = Geom::Point3d.new.offset v1
    a*= -1 if (negAnglePointCheck.distance negAnglePointNeg) < (negAnglePointCheck.distance negAnglePointPos)

    a

  end

  def self.calc_path(controls, curve_algorithm, segments = 32)
    #Returns path as array of Point3d objects
    #Called in @controls, @segments and @curve_algorithm attribute writer to always keep @path updated.
    #Also called in track insert and positioning tools for previews.
    
    #All paths end segments are parallel to corresponding control vector in x-y plane

    #For historical reasons named variables are used instead of array elements for controls.
    #This is however easier to read and might be kept permanently
    point_start = controls[0]
    point_end = controls[1]
    vector_start = controls[2]
    vector_end = controls[3]
    
    if point_start == point_end && vector_start.parallel?(vector_end)
      raise ArgumentError, "Cannot calculate path for 0 length track."
      return
    end

    if is_straight? controls
      #Straight track
      #Hooray! Easy to create path!

      path = [point_start, point_end]
    else
      #Curved track
      #Oh dear...

      #Multiple curve algorithms to chose from.
      #Different algorithms have different purposes. Some include smooth ends of the curve while others doesn't.
      #If better algorithms are coded user can still redraw track to the old path so track surroundings doesn't break.

      case curve_algorithm
      when "hard_corner"
        #Add corner where lines defined by end points and end vectors intersect
        # * Not using length of vectors, just direction
        # - Elevation not calculated from vectors
        # - Not a curve

        path = [point_start]
        #Get intersection from planes since the ends might have different z
        plane_start = [point_start, vector_start.cross(Z_AXIS)]
        plane_end = [point_end, vector_end.cross(Z_AXIS)]
        line_between = Geom.intersect_plane_plane(plane_start, plane_end)
        if line_between#Vertical line
          #Not straight in x-y plane
          point_between = line_between[0]
          point_between.z = (point_start.z + point_end.z)/2
          path << point_between
        end
        path << point_end

      when "c_bezier"
        #Cubic bezier
        # * Smooth endings, can make up a whole curve
        # + Elevation calculated from vectors.
        # + must not be symmetric

        #bezier path coded "backwards"

        #Place control points on point.offset vector
        p0 = point_end
        p1 = point_end.offset vector_end
        p2 = point_start.offset vector_start
        p3 = point_start

        #Segments must be float for division to work
        segments = Float(segments)

        path = [p3]

        0.upto(segments-1) do |i|
          t = i+0.5
          q0 = Geom::Point3d.linear_combination t/segments, p0, (segments-t)/segments, p1
          q1 = Geom::Point3d.linear_combination t/segments, p1, (segments-t)/segments, p2
          q2 = Geom::Point3d.linear_combination t/segments, p2, (segments-t)/segments, p3

          r0 = Geom::Point3d.linear_combination t/segments, q0, (segments-t)/segments, q1
          r1 = Geom::Point3d.linear_combination t/segments, q1, (segments-t)/segments, q2

          b = Geom::Point3d.linear_combination t/segments, r0, (segments-t)/segments, r1
          path << b
        end#each
        path << p0

        #Make end segments follow control vectors
        #Fit second and second to last point to line defined by corresponding endpoint and its vector
        help_line_start = [point_start, vector_start]
        help_line_end = [point_end, vector_end]

        second_segment_line = [path[1], (path[2] - path[1])]
        second_to_last_segment_line = [path[-2], (path[-3] - path[-2])]

        second_point = Geom.closest_points(help_line_start, second_segment_line)[0]
        second_to_last_point = Geom.closest_points(help_line_end, second_to_last_segment_line)[0]

        path[1] = second_point
        path[-2] = second_to_last_point

      when "arc"
        #Arc
        # * No smooth endings, multiple identical tracks can make up bigger curve when simulating model railway (also good in industrial areas I think)
        # * Not using length of vectors, just direction
        # - Elevation not calculated from vectors.
        # - must be symmetric

        #Get center axis
        plane_start = [point_start, MyGeom.flatten_vector(vector_start)]
        plane_end = [point_end, MyGeom.flatten_vector(vector_end)]

        if plane_start[1].parallel?(plane_end[1]) && plane_start[0].on_plane?(plane_end)
          #Planes are the same meaning angle is 180 degrees.
          #point_center is between start and end

          point_center = Geom::Point3d.linear_combination 0.5, point_start, 0.5, point_end

        elsif plane_start[1].parallel? plane_end[1]
          #Planes are co-planar but not the same
          #Draw straight track

          return [point_start, point_end]

        else
          #Planes intersect
          #Track turns any angle but +-180 degrees

          line_intersect = Geom.intersect_plane_plane plane_start, plane_end
          point_center = line_intersect[0]

        end

        #Get angle
        angle_total = MyGeom.angle_in_plane(point_start-(point_center), point_end-(point_center)).abs
        #Explementary angle if start vector points away from end
        angle_total = 2*Math::PI - angle_total if point_start.offset(vector_start).distance(point_end) > point_start.offset(vector_start).distance(point_start.offset(point_start-point_end))
        #Distinguish cc from cw (+180 degrees and -180 degrees differs)
        angle_total*= -1 if MyGeom.flatten_vector(vector_start).cross(MyGeom.flatten_vector(point_end-(point_start))).z < 0

        angle_per_segment = angle_total/segments

        #Radius
        #Arc start and ends with a half segment so extrusion face is perpendicular to path and arc can continue smoothly in other tracks
        point_center.z = point_start.z
        radius = point_start.distance point_center

        radius/= Math.cos(angle_per_segment/2)
        vector_start_from_center = (point_start - point_center)
        vector_start_from_center.length = radius
        point_template = point_center.offset vector_start_from_center

        #Create path
        path = [point_start]
        0.upto(segments-1) do |n|
          #Angle, start with half segment
          angle = (0.5+n)*angle_per_segment

          #Rotate point
          trans_rotate = Geom::Transformation.rotation(point_center, Z_AXIS, angle)
          point = point_template.transform(trans_rotate)

          #Height, linear
          point.z = point_start.z + (point_end.z - point_start.z)*((0.5+n)/segments)

          path << point
        end#each
        path << point_end
      end#case
    end#if eslif

    #Return
    path

  end

  def self.calc_trans_along(controls, curve_algorithm, distance)
    #Calculate numerous transformation along track and return as array of Transformation objects.
    #Distance is approximate and rounded so the number of transformations is an integer.
    
    #Used to place ties for instance.
    #The end transformations are positioned half a "distance" in from track ends.
    #Calculates transformations algebraically for arcs, not based on polyline approximation.

    start_vector_flat = MyGeom.flatten_vector(controls[2], [Geom::Point3d.new, Z_AXIS])
    trans_start = Geom::Transformation.axes controls[0], (start_vector_flat).cross(Z_AXIS), start_vector_flat, Z_AXIS

    trans_a = []

    if is_straight? controls
      #Straight track

      #Adjust distance to fit total length
      length = controls[0].distance controls[1]
      return [] if length < distance
      number_of_distances = (length/distance).round
      distance = length/number_of_distances

      #Create transformations
      vector_offset = controls[2].clone
      0.upto(number_of_distances-1) do |i|
        #First transformation is a half distance from end

        vector_offset.length = (0.5 + i) * distance
        point = controls[0].offset vector_offset

        trans_a << Geom::Transformation.axes(point, (start_vector_flat).cross(Z_AXIS), start_vector_flat, Z_AXIS)

      end

    else
      #Curved track

      case curve_algorithm
      when "hard_corner"
        #Add corner where x-y lines defined by end points and end vectors intersect

        #Get intersection from planes since the ends might have different z
        plane_start = [controls[0], controls[2].cross(Z_AXIS)]
        plane_end = [controls[1], controls[3].cross(Z_AXIS)]
        line_between = Geom.intersect_plane_plane(plane_start, plane_end)
        if line_between

          #Not straight in x-y plane
          point_between = line_between[0]
          point_between.z = (controls[0].z + controls[1].z)/2

          points = [controls[0], point_between, controls[1]]
          0.upto(1) do |i|
            #Adjust distance to fit total length
            length = points[i].distance points[i+1]
            return [] if length < distance
            number_of_distances = (length/distance).round
            distance = length/number_of_distances

            vector_along = MyGeom.flatten_vector points[i+1].-(points[i])

            #Create transformations
            vector_offset = points[i+1] - points[i]
            0.upto(number_of_distances-1) do |j|
              #First transformation is a half distance from end

              vector_offset.length = (0.5 + j) * distance
              point = points[i].offset vector_offset

              trans_a << Geom::Transformation.axes(point, (vector_along).cross(Z_AXIS), vector_along, Z_AXIS)
            end
          end#each

        else
          #Straight in x-y plane

          #Adjust distance to fit total length
          length = controls[0].distance controls[1]
          return [] if length < distance
          number_of_distances = (length/distance).round
          distance = length/number_of_distances

          #Create transformations
          vector_offset = controls[1] - controls[0]
          0.upto(number_of_distances-1) do |i|
            #First transformation is a half distance from end

            vector_offset.length = (0.5 + i) * distance
            point = controls[0].offset vector_offset

            trans_a << Geom::Transformation.axes(point, (start_vector_flat).cross(Z_AXIS), start_vector_flat, Z_AXIS)
          end
        end

      when "c_bezier"
        #Really really hard to do algebraic.
        #Do 'filthy' approximation instead.

        path = MyGeom.calc_path controls, curve_algorithm
        trans_a = MyGeom.calc_trans_along_path path, distance

      when "arc"

        #Get center axis
        plane_start = [controls[0], MyGeom.flatten_vector(controls[2])]
        plane_end = [controls[1], MyGeom.flatten_vector(controls[3])]

        if plane_start[1].parallel?(plane_end[1]) && plane_start[0].on_plane?(plane_end)
          #Planes are the same meaning angle is 180 degrees.
          #point_center is between start and end

          point_center = Geom::Point3d.linear_combination 0.5, controls[0], 0.5, controls[1]

        elsif plane_start[1].parallel? plane_end[1]
          #Planes are co-planar but not the same
          #Draw straight track

          #Adjust distance to fit total length
          length = controls[0].distance controls[1]
          return [] if length < distance
          number_of_distances = (length/distance).round
          distance = length/number_of_distances

          #Create transformations
          vector_offset = controls[1] - controls[0]
          0.upto(number_of_distances-1) do |i|
            #First transformation is a half distance from end

            vector_offset.length = (0.5 + i) * distance
            point = controls[0].offset vector_offset

            trans_a << Geom::Transformation.axes(point, start_vector_flat.cross(Z_AXIS), start_vector_flat, Z_AXIS)

          end

          return trans_a

        else
          #Planes intersect
          #Track turns any angle but +-180 degrees

          line_intersect = Geom.intersect_plane_plane plane_start, plane_end
          point_center = line_intersect[0]

        end

        #Get radius
        radius = controls[0].distance_to_line [point_center, Z_AXIS]

        #Get angle
        angle_total = MyGeom.angle_in_plane(controls[0]-point_center, controls[1]-point_center).abs
        #Explementary angle if start vector points away from end
        angle_total = 2*Math::PI - angle_total if controls[0].offset(controls[2]).distance(controls[1]) > controls[0].offset(controls[2]).distance(controls[0].offset(controls[0]-controls[1]))
        #Distinguish cc from cw (+180 degrees and -180 degrees differs)
        angle_total*= -1 if MyGeom.flatten_vector(controls[2]).cross(MyGeom.flatten_vector(controls[1]-controls[0])).z < 0

        #Get angle between each object
        length = radius * angle_total
        number_of_distances = (length/distance).round.abs
        distance = length/number_of_distances
        return [] if length.abs < distance.abs
        angle_per_distance = distance/radius

        #Create transformations
        0.upto(number_of_distances-1) do |n|
          #Angle, start half distance from end
          angle = (0.5+n)*angle_per_distance

          #Height, linear
          move_up = Z_AXIS.clone
          move_up.length = (controls[1].z - controls[0].z)*((0.5+n)/number_of_distances)
          trans_lift = Geom::Transformation.translation move_up

          #Rotation transformation
          trans_rotate = Geom::Transformation.rotation(point_center, Z_AXIS, angle)

          trans_a << trans_rotate*(trans_lift*(trans_start))

        end#each
      end

    end

    trans_a

  end

  def self.calc_trans_along_path(path, distance, start_at_distance = nil)
    #Calculate numerous transformation along path and return as array of Transformation objects.
    #Path is an array of points
    #Distance is approximate and rounded so the number of transformations is an integer.
    #Optional start_at_distance (default nil) places last and first transformation a whole distance in. Otherwise a half is used.
    
    #Used to place structures along tack.
    #Calculates transformations numerically since the method doesn't know what algorithm was used to calculate path and
    #path can even span several tracks with different algorithms.
    
    #NOTE: FEATURE: argument for set number of elements. argument for zig zag. add those parameters to filenames. let structure properties override it.
    
    path_length = self.path_length path
    return [] if path_length < distance
    
    number_of_distances = (path_length/distance).round
    distance = (path_length/number_of_distances).to_l

    #First object is always half a distance from track end
    distance_to_next = distance
    distance_to_next *= 0.5 unless start_at_distance

    #This var must be defined before refereed to (in the order the parser reads the file, no the order it's executed)
    p_mid_next = nil
    
    #Segment counter
    i_seg = 0
    
    trans_a = []

    #Follow path to end
    #Loop segments
    until i_seg == path.length-1

      p_start = path[i_seg]
      p_end = path[i_seg+1]
      p_next = path[i_seg+2]
      v_seg = p_end - p_start
      l_seg = p_end.distance p_start
      p_offset_start = p_start
      
      point = p_start#point to offset trans origin from

      #Vectors and midpoints of this, previous and next segment.
      #Used to get approximated vector for smooth curve used for transformations
      if i_seg == 0
        #First segment, use start as midpoint since it's really a half segment
        p_mid = p_start
      else
        #Not first segment, use values from previous segment
        p_mid_prev = p_mid
        p_mid = p_mid_next
      end
      if i_seg < path.length-2
        #Not second to last (or last) segment, get midpoint of next segment
        p_mid_next = Geom.linear_combination 0.5, p_end, 0.5, p_next
      else
        #Second to last segment (or last), use end as midpoint since it's really a half segment
        p_mid_next = path[-1]
      end

      #Loop transformations at this segments
      while distance_to_next < point.distance(p_end)
        #There is at least an other transformation within this segment

        #Find point
        l_o = distance_to_next
        point = p_offset_start.offset v_seg, l_o
        
        #Get curve tangent approximation
        l_mid_prev = point.distance p_mid_prev if p_mid_prev
        l_mid = point.distance p_mid
        l_mid_next = point.distance p_mid_next
        if l_mid_prev && l_mid_prev < l_mid_next
          p_mid_other = p_mid_prev
          v_other = p_start - p_mid_prev
          l_mid_other = l_mid_prev
        else
          p_mid_other = p_mid_next
          v_other = p_mid_next - p_end
          l_mid_other = l_mid_next
        end
        influence = 0.5*(l_mid/l_mid_other)
        influence = 0.5 if influence > 0.5#It shouldn't be possible for this to be greater than 0.5 but sometimes it still is.
        v_smooth = Geom.linear_combination 1.0-influence, v_seg, influence, v_other
        #ents = Sketchup.active_model.active_entities.add_group.entities
        #ents.add_line(point, point.offset(v_seg, 1.m)).material = Sketchup::Color.new "Red" if v_seg.valid?
        #ents.add_line(point, point.offset(v_smooth, 1.m)).material = Sketchup::Color.new "Green" if v_smooth.valid?
        #ents.add_line(point, point.offset(v_other, 1.m)).material = Sketchup::Color.new "Blue" if v_other.valid?
        #ents.add_text influence.to_s, point
        
        #trans = Geom::Transformation.axes point, v_seg.cross(Z_AXIS), v_seg, Z_AXIS
        trans = Geom::Transformation.axes point, v_smooth.cross(Z_AXIS), v_smooth, Z_AXIS
        trans_a << trans

        #"Counter"
        distance_to_next = distance
        p_offset_start = point

      end

      i_seg += 1
      distance_to_next -= point.distance p_end
      
    end

    trans_a
    
  end

  def self.calc_trans_ends(controls_or_path)  
    #Calculate transformations to place stuff at track/path ends.
    
    #Input can either be a controls array (start point, end point, start vectir, end vector)
    #or an existing path.
    
    #Returns array of 4 transformations.
    #0 and 2 is for track/path start.
    #1 and 3 is for track/path end.
    #All have z pointing upwards and y towards track/path.
    #3 first have x pointing right when standing at track/path end and looking at it.
    #Last one is mirrored in y so it can be used for asymmetric parts such as double track bridge heads.
    #0 and 2 are the same but occurs twice in the array so the index can be <path end(0|1) + mirror?(0|2)>.
    
    if controls_or_path.all?{ |i| i.class == Geom::Point3d }
      #Input is path
      p_start = controls_or_path[0]
      p_end = controls_or_path[-1]
      v_start = controls_or_path[1] - p_start
      v_end = controls_or_path[-2] - p_end
    elsif controls_or_path.length == 4 && controls_or_path[0..1].all?{ |i| i.class == Geom::Point3d } && controls_or_path[2..3].all?{ |i| i.class == Geom::Vector3d }
      #Input is controls array
      p_start, p_end, v_start, v_end = controls_or_path
    else
      raise ArgumentError, "Argument must either be an array of Point3d objects (a path) or an array with 2 Point3d objects and 2 Vector3d objects (controls)."
      return
    end

    v_start = MyGeom.flatten_vector v_start
    v_end = MyGeom.flatten_vector v_end

    trans = []
    
    trans << Geom::Transformation.axes(p_start, v_start*Z_AXIS, v_start, Z_AXIS)
    trans << Geom::Transformation.axes(p_end, v_end*Z_AXIS, v_end, Z_AXIS)

    trans << trans[0]
    trans << Geom::Transformation.axes(p_end, Z_AXIS*v_end, v_end, Z_AXIS)

    trans

  end

  def self.controls_offset(controls_old, distance)
    #Find controls for offset track.
    #Offset to left seen from track start end.
    
    controls = controls_old.map{ |i| i.clone }
    
    #Move points
    v_offset = Z_AXIS*flatten_vector(controls[2])
    controls[0] = controls[0].offset v_offset, distance
    v_offset = flatten_vector(controls[3])*Z_AXIS
    controls[1] = controls[1].offset v_offset, distance
    
    #Vector length (only affects Beziers)
    #Simply move inner control points along bisectris of their corner in polyline.
    middle_vector = controls_old[0].offset(controls_old[2]) - controls_old[1].offset(controls_old[3])#Vector between inner control points
    plane0 = [controls_old[0].offset(controls_old[2]), Geom.linear_combination(0.5, controls_old[2], 0.5, middle_vector.reverse)]
    plane1 = [controls_old[1].offset(controls_old[3]), Geom.linear_combination(0.5, controls_old[3], 0.5, middle_vector)]
    
    controls[2].length = controls[0].distance(Geom.intersect_line_plane([controls[0], controls[2]], plane0)) if plane0[1].valid? && controls[2].valid?
    controls[3].length = controls[1].distance(Geom.intersect_line_plane([controls[1], controls[3]], plane1)) if plane1[1].valid? && controls[3].valid?
    
    controls
      
  end
  
  def self.curve(p_prev, p, p_next, r, k)
    #Create composite or wholly curve.

    #p_prev is a point before the curve.
    #p is a where track into curve and out of curve would intersect if there was no curve.
    #p_next is a point after the curve.
    #r is the maximum radius.
    #k is the coefficient of curvature to length.
    
    #Returns array of track data making up the curve,
    #3 for composite curve and 2 for wholly curve.
    
    
    ##Test code for simple arc
    #
    #p_others = [p_prev, p_next]
    #
    ##Angle in horizontal plane
    #angle_h = MyGeom.angle_in_plane(p-p_prev, p-p_next).abs
    #
    ##Get controls for curve
    #c_points = []
    #c_vectors = []
    #0.upto(1){ |j|
    #  #Horizontal distance between node and j. end of curve.
    #  node_2_curve_end_h = (r/Math.tan(angle_h/2)).to_l
    #  
    #  v = p - p_others[j]
    #  a_grad = MyGeom.gradient v
    #  node_2_curve_end = node_2_curve_end_h/Math.cos(a_grad)
    #  
    #  v.length = node_2_curve_end
    #  p_c = p.offset(v.reverse)
    #  c_points << p_c
    #  c_vectors << v
    #CLSOINGBRACKET
    #controls = c_points + c_vectors
    #
    #return [{ :controls => controls, :curve_algorithm => "arc" }]
    
    
    #Angle in horizontal plane
    angle_h = MyGeom.angle_in_plane(p-p_prev, p_next-p)
    
    #Calculate transition curve with half total angle as max-angle and r as max-radius.
    trans_curve = curve_transition k, r, angle_h.abs*0.5
    
    #Length in horizontal plane between node and curve ends
    length_h = trans_curve[:end_center].x + trans_curve[:end_center].y*Math.tan(angle_h.abs*0.5)

    v_prev = p_prev - p
    v_next = p_next - p
    v_prev.z = v_next.z = 0
    
    c_start = p.offset v_prev, length_h
    c_end = p.offset v_next, length_h
    
    trans_start = Geom::Transformation.axes c_start, v_prev.reverse, (angle_h<0 ? v_prev*Z_AXIS : v_prev*Z_AXIS.reverse)
    trans_end = Geom::Transformation.axes c_end, v_next.reverse, (angle_h<0 ? v_next*Z_AXIS.reverse : v_next*Z_AXIS)
    
    track_data = []
    
    track_data << {
      :controls => trans_curve[:controls].map { |i| i.transform trans_start },
      :curve_algorithm => "c_bezier"
    }
    track_data << {
      :controls => trans_curve[:controls].map { |i| i.transform trans_end },
      :curve_algorithm => "c_bezier"
    }
    
    #Reverse controls of last curve so controls[0] is towards p_prev and controls[1] towards p_next
    c = track_data[-1][:controls]
    c[0], c[1], c[2], c[3] = c[1], c[0], c[3], c[2]
    
    #Add arc if transition curves don't meet
    if track_data[0][:controls][1] != track_data[1][:controls][0]#Previous check of trans_curve[:angle] != angle_h.abs*0.5 sometimes returned true even for wholly curves :(
      controls = [
        track_data[0][:controls][1],
        track_data[1][:controls][0],
        track_data[0][:controls][3].reverse,
        track_data[1][:controls][2].reverse
      ]
      track_data.insert(-2, { :controls => controls, :curve_algorithm => "arc"})
    end
    
    #Project coordinates to plane (until now they've been in horizontal plane)
    plane = [p, (p_prev-p)*(p_next-p)]
    track_data.each do |t|
      t[:controls] = t[:controls].map { |i| lift_to_plane i, plane }
    end
    
    #return array of track data (controls + curve algorithm)
    track_data
  
  end
  
  def self.curve_transition(k, r_max = nil, angle_max = nil)
    #Calculate cubic parabola transition curve.
    #In theory a Euler spiral is the best transition curve with a curvature proportional to arc length.
    #however it's difficult to calculate.
    #Cubic parabolas has a curvature that is proportional to the projected length on a straight line which is close enough.
    
    #r_max is the radius where curve should stop unless curve reaches angle_max before. [inches]
    #k is proportionality constant between length and curvature. [m^-2]
    #angle_max is the angle where curve should stop unless curve reaches r_max before. [radians]
    
    #Returns controls array [start point, end point, start vector, end vector] for bezier curve
    #angle (radians) and end radius.
    #Coordinates are relative to origin being the straight end and curve being in first quadrant.
    
    #Convert k from m^2 to inch^-2
    k = k.to_m.to_m
    
    #Angle cannot be greater than 24 degrees for a cubic parabola to work as an approximation of an euler spiral.
    angle_max_parabola = 24.degrees
    
    #Length (projected) from angle
    angle_max = [angle_max_parabola, angle_max].compact.min
    x = Math.sqrt(Math.tan(angle_max)/(3*k))#Only interesting in positive solution
    
    end_radius = (Math.sqrt((1+9*k**2*x**4)**3)/(6*k*x)).to_l
    
    #If end radius is to big. limit length after r_max.
    #Numeric solution
    if r_max && end_radius < r_max
      end_radius = r_max
      x_min = 1
      x_max = x
      r_test = nil
      x_test = nil
      15.times do
        x_test = (x_min+x_max)*0.5
        r_test = (Math.sqrt((1+9*k**2*x_test**4)**3)/(6*k*x_test)).to_l
        if r_test < r_max
          x_max = x_test
        else
          x_min = x_test
        end
      end
      x = x_test
      #puts "Desired: #{r_max}, Achieved: #{r_test}, Difference: #{(r_max-r_test).abs.to_l}"
    end
        
    #Points
    p_start = ORIGIN.clone
    y = k*x**3
    z = 0
    p_end = Geom::Point3d.new x, y, z
    
    #Vectors
    y_prim = k*3*x**2
    v_start = Geom::Vector3d.new 1, 0, 0
    v_end = Geom::Vector3d.new -1, -y_prim, 0
    
    #Vector lengths (to make cubic bezier curve)
    bezier_cuad_c = Geom::Point3d.new(x/2, y_prim*x/2)
    v_start.length = p_start.distance(bezier_cuad_c)*2.0/3.0
    v_end.length = p_end.distance(bezier_cuad_c)*2.0/3.0
    
    controls = [p_start, p_end, v_start, v_end]
    
    end_angle = Math.atan(y_prim)
    end_center = p_end.offset v_end*Z_AXIS, end_radius
    
    #Return
    { :controls => controls, :angle => end_angle, :end_radius => end_radius, :end_center => end_center}
  
  end
  
  def self.point_between_points?(point_between, p1, p2)
    #Check if point is between other points
    #(closer to both of them than they are to each other)
    
    reference_distance = p1.distance p2
    return false if point_between.distance(p1) > reference_distance
    return false if point_between.distance(p2) > reference_distance
    
    true
  
  end
  
  def self.flatten_vector(vector, plane = nil)
    #Flatten vector to plane
    #plane's point does not affect output, only its normal does
    #Use x-y plane if no plane is set

    plane = PLANE_HORIZONTAL unless plane

    pointStart = plane[0]
    normal = plane[1]

    #Point at end of vector
    pointVectorEnd = pointStart.offset vector
    #point projected to plane
    pointIntesection = pointVectorEnd.project_to_plane plane
    #vector from start point to intersection
    outputVector = pointIntesection - pointStart

    outputVector

  end

  def self.gradient(v)
    #Get gradient angle from vector.
    #Angle in vertical plane
    
    delta_z = v.z
    v = v.clone
    v.z = 0
    delta_h = v.length
    
    Math.atan(delta_z/delta_h)
    
  end
  
  def self.intersect_line_sphere(line, center, radius)
    #Get intersection points between line and sphere
    #Returns array of points. empty array on failure

    lineStart = line[0]
    lineVector = line[1].normalize
    
    unless lineVector.valid?
      raise ArgumentError.new("Zero length vector can't be used to intersect sphere.")
    end
    
    if radius == 0
      return [] unless center.on_line? line
      return [center]
    end#if

    #Calculate distance from line's start along line to intersections

    firstTerm = -(lineVector.dot(lineStart-(center)))
    secondTermSquared = (lineVector.dot(lineStart-(center)))**2 - (lineStart-(center)).dot(lineStart-(center)) + radius**2

    #If root is imaginary the line does not intersect the sphere
    return [] if(secondTermSquared < 0)

    secondTerm = Math.sqrt(secondTermSquared)

    solution0 = firstTerm+secondTerm
    solution1 = firstTerm-secondTerm

    #Find points
    point0 = lineStart.offset lineVector, solution0
    point1 = lineStart.offset lineVector, solution1

    #return points
    [point0, point1]

  end

  def self.identity_matrix?(t)
  
    a = t.to_a
    a_ref = Geom::Transformation.new.to_a
    a.each_with_index do |n, i|
      return false unless n.to_l == a_ref[i].to_l
    end
    
    true
    
  end
  
  def self.is_straight?(controls)
  
    line = [controls[0], controls[2]]
    #vector_start.reverse.samedirection?(vector_end) && point_end.on_line?([point_start, vector_start])#Precision problem with tracks really freaking close to be straight but on_line? returns false. These tracks are too straight to be drawn with bezier algorithm.
    
    controls[2].reverse.samedirection?(controls[3]) && controls[1].distance_to_line(line) < 0.01
    
  end
  
  def self.lift_to_plane(point_or_vector, plane, vector = nil)
    #Move point or vector tip along given vector onto a plane
  
    plane ||= PLANE_HORIZONTAL
    vector ||= Z_AXIS
    
    #Convert vector to point for consistent code
    is_vector = point_or_vector.class == Geom::Vector3d
    point_or_vector = plane[0].offset point_or_vector if is_vector
    
    line = [point_or_vector, vector]
    point_or_vector = Geom.intersect_line_plane line, plane
    
    #Convert back to vector if needed
    point_or_vector = point_or_vector - plane[0] if is_vector
    
    point_or_vector
  
  end
  
  def self.move_group_origin group, trans_new
    #Moves the axis of the group without moving content compared to global axis
    
    trans_old = group.transformation
    
    group.name += ""#Group.make_unique is deprecated but changing name seems to work too
    
    #Move group
    group.transformation = trans_new
    
    #Move back group content
    trans_change = trans_new*(trans_old.inverse)
    trans_change = trans_new.inverse*trans_change*trans_new#Transform transformation so it's relative to the new group transformation and not model axis
    group.entities.transform_entities trans_change.inverse, group.entities.to_a
  
    true
    
  end
  
  def self.path_length(path)

    (0..path.length-2).map { |i| path[i].distance path[i+1] }.inject { |sum, n| sum + n }.to_l
  
  end
  
  def self.path_truncate(old_path, length)
  
    old_length = self.path_length old_path
    return old_path if old_length <= length
    remaining_length = length
    
    new_path = [old_path[0]]
    0.upto(old_path.length-2) do |i|
      p_start = old_path[i]
      p_end = old_path[i+1]
      segment_length = p_start.distance(p_end)
      
      if segment_length < remaining_length
        remaining_length -= segment_length
        new_path << p_end
        next
      end
      
      segment_vector = p_end - p_start
      new_path << p_start.offset(segment_vector, remaining_length)
      break
    end
    
    new_path
    
  end

end# Module

end# Module
