# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

# Internal: Various draw methods for View.
module MyView

  # Highlight a plane by drawing a cross in it.
  #
  # view   - The View to draw to.
  # plane  - The plane to display, formated as array of Point and normal Vector.
  # radius - How far from point lines should reach.
  #
  # Returns nothing.
  def self.draw_plane_outline(view, plane, radius)
    
    # Pattern to highlight plane.
    points = [
      Geom::Point3d.new(radius, 0, 0), Geom::Point3d.new(-radius, 0, 0),
      Geom::Point3d.new(0, radius, 0), Geom::Point3d.new(0, -radius, 0)
    ]
    
    # Rotate
    unless plane[1].parallel? Z_AXIS
      rotation_vector = plane[1] * Z_AXIS
      angle = plane[1].angle_between Z_AXIS
      angle = Math::PI - angle
    
      trans  = Geom::Transformation.rotation ORIGIN, rotation_vector, angle
      points.each { |p| p.transform! trans }
    end
    
    # Translate
    trans  = Geom::Transformation.translation plane[0]
    points.each { |p| p.transform! trans }
    
    # Draw
    point_temp = Geom::Point3d.new.offset! plane[1]
    view.set_color_from_line ORIGIN, point_temp
    view.draw_lines points
    
    nil
    
  end

  # Draws an arrow head pointing in given direction.
  #
  # view     - The View to draw to.
  # point    - Point where arrow head should be located.
  # vector   - Which way arrow head should point.
  # centered - Center arrow head at point instead of having tip on it
  #            (default: false).
  #
  # Returns nothing.
  def self.draw_arrow(view, point, vector, radius, centered = false)
    
    return unless vector.valid?
    
    vector_forward = vector.clone
    vector_forward.length = radius*0.5
    vector_backward = vector_forward.reverse
    vector_side1 = vector_forward*Z_AXIS
    vector_side1.length = radius
    vector_side2 = vector_side1.reverse
    
    unless centered
      vector_forward.length = 0
      vector_backward.length = radius
    end
    
    point_front = point.offset vector_forward
    point_back = point.offset vector_backward
    point_side1 = point_back.offset vector_side1
    point_side2 = point_back.offset vector_side2

    view.draw_polyline point_side1, point_front, point_side2
    
    nil
    
  end

  # Draw 3d polyline on top of all entities instead of in 3d context as
  # View.draw_polyline does.
  #
  # view - The View to draw to.
  # path - Array of Point3d object.
  #
  # Returns nothing.
  def self.draw_polyline_on_top(view, path)
    
    path2d = path.map { |p| view.screen_coords p }
    view.draw2d GL_LINE_STRIP, path2d
    
    nil
    
  end

end #Module

end #Module