# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

# Internal: Wrapper for all observer related code in plugin.
# Concentrating observers here that actually has to do with different classes
# still is cleaner and easier to maintain than spreading out observers and
# code attaching them to new models.
module Observers

  # Whether observers are temporarily disabled or not.
  @@disabled = false
  
  # Whether currently executed code was called from observer fired by Undo or
  # Redo. This should suppress all drawing.
  @@undo_redo = false

  # Accessors/Macros.
  
  def self.disabled?; @@disabled; end
  def self.disabled=(v); @@disabled = v; end
  def self.disable; @@disabled = true; end
  def self.enable; @@disabled = false; end
    
  def self.undo_redo?; @@undo_redo; end
  def self.undo_redo=(v); @@undo_redo = v; end
   
  # Methods to be called from inside module.

  def self.init_model(model)
    #Runs when a new model is loaded or created, or sketchup starts.
    #Also runs when this file is reloaded
    #Purges various objects not belonging to an open model.
    #Initialize objects from attributes in new model.
    #Does not initialize objects that already have been initialized.

    #Track and rolling stock objects aren't saved with the model, but the groups they are drawn in holds their data.
    #This is because Sketchup::Group cannot be saved to an [model] attribute and the linked group would be lost from the object if trying to do so.
    #Recreate objects from groups when loading model.

    #Unload (purge) track, rolling stock, train and balise objects if their model is no longer loaded.
    to_delete = []
    Track.instances.each  { |t|  to_delete << t if t.group.deleted? }  # Also deletes balise objects linked to the track.
    RStock.instances.each { |rs| to_delete << rs if rs.group.deleted? }# Also removes train when it's empty.
    to_delete.each        { |i|  i.delete! }

    # Animation is stopped for inactive models from the frame method of the animate class

    #Initialize track objects from groups
    initialized_tracks = []
    model.entities.each do |i|
      next unless i.class == Sketchup::Group
      next unless Track.group_is_track? i
      next if Track.get_from_group i
      initialized_tracks << Track.new(i)
    end

    #Find track connections
    Track.find_all_connections initialized_tracks
    
    #Load balise objects
    Balise.load_from_file model

    #Initialize rolling stock objects from groups
    #Must be done in its own loop after tracks are initialized so rolling stocks can be assigned the track they stand on
    initialized_r_stocks = []
    model.entities.each do |i|
      next unless RStock.group_is_r_stock? i
      next if RStock.get_from_group i
      r_stock = RStock.new(i)
      r_stock.find_tracks
      initialized_r_stocks << r_stock
    end

    #Make trains out of rolling stocks.
    #Rolling stocks have train_name and train_position saved to them

    #Group rolling stocks into trains
    trains = []
    until initialized_r_stocks.empty?
      train_name = initialized_r_stocks[0].train_name
      r_stocks_in_train = initialized_r_stocks.select { |rs| rs.train_name == train_name}
      trains << r_stocks_in_train
      initialized_r_stocks-= r_stocks_in_train
    end
    #Initialize train objects after sorting rolling stocks in them
    trains.each do |train_array|
      train_array = train_array.sort_by { |rs| rs.train_position }# No sort_by! for ruby 1.8.6.
      Train.new train_array
    end
    
    #Add observer to structures
    #The structure objects do not exist when not used, unlike tracks that are loaded on model load.
    #Tracks and RTocks get their observers when initialized.
    model.entities.each do |i|
      next unless Structure.group_is_structure? i
      i.add_observer MyInstanceObserver.new
    end

    #Add model observer
    model.add_observer MyModelObserver.new

    #Add entities observer to model.entities
    model.entities.add_observer MyEntitiesObserver.new

    #Initialize animation
    Animate.new model

  end

  # Observer classes.
  
  class MyAppObserver < Sketchup::AppObserver

    def onOpenModel(mod)
      #Initialize railroads when opening model from within Sketchup
      return if Observers.disabled?
      Observers.init_model(mod)
    end

    def onNewModel(mod)
      #Initialize railroads when creating new model.
      return if Observers.disabled?
      Observers.init_model(mod)
    end

  end#class

  class MyModelObserver < Sketchup::ModelObserver

    def onPreSaveModel(model)
      return if Observers.disabled?
    
      #Save rolling stock variables as group attributes so the values can be retrieved when reopening the application.
      RStock.instances.each { |rs| rs.save_attributes }
      
      #Save balise objects to file
      Balise.save_to_file model
    end
    
    def onTransactionUndo(model)
    
      return if Observers.disabled?
      
      # Set flag telling code executed now is called from an Undo event. 
      Observers.undo_redo = true
      UI.start_timer(0, false){ Observers.undo_redo = false }
      
      #Compare track attributes with their groups attributes, update track object when it differs from group.
      Track.update_from_groups
      
    end
    
    def onTransactionRedo(model)
    
      return if Observers.disabled?
      
      # Set flag telling code executed now is called from an Undo event. 
      Observers.undo_redo = true
      UI.start_timer(0, false){ Observers.undo_redo = false }
      
      # Compare track attributes with their groups attributes, update track object when it differs from group.
      Track.update_from_groups
            
    end

  end#class

  class MyEntitiesObserver < Sketchup::EntitiesObserver

    def onElementAdded(enteties, entity)
      #New element added
      #Also fires after undoing deletion
      #If it's a track group being copied, initialize track object & update connected tracks

      return if Observers.disabled?
      Observers.disable
      
      if Track.group_is_track? entity
        puts "Track added (onElementAdded, Undo/Redo: #{Observers.undo_redo?})."
        #Track added (copied in).
        
        t = Track.new(entity)
        
        # Find connections and add tracks that lost or got connection to this to drawing queue.
        # Redraw ends of new track since buffers may differ from track it was copied from.
        changed = t.update_connections
        unless Observers.undo_redo?
          Track.drawing_queue_ends += changed
          t.queue_end_drawing
          UI.start_timer(0, false) { Track.draw_queued(true) }
        end

        # Add observers
        entity.add_observer(MyInstanceObserver.new)
        entity.add_observer(MyEntityObserver.new)
        
      elsif RStock.group_is_r_stock? entity
        puts "Rolling stock added (onElementAdded, Undo/Redo: #{Observers.undo_redo?})."
        #Rolling stock added (copied into model)
        #This shouldn't be called from creating new rolling stocks since the group is added without attributes and attributes then added to it
        
        #Initialize new rolling stock and train for it
        r_stock = RStock.new entity
        r_stock.find_tracks
        name = Train.unique_name(r_stock.train_name)#Make train name unique before new train is initialized
        r_stock.train_name = name
        train = Train.new [r_stock]
        
      end

      Observers.enable
      
    end

    def onElementModified(entities, entity)
      #Fires when track group itself is modified (e.g. scaled or moved),
      #or its content entities, such as when any draw function is called

      return if Observers.disabled?
      Observers.disable
      
      if Track.group_is_track? entity
        puts "Track modified (onElementModified, Undo/Redo: #{Observers.undo_redo?})."
        #Track changed (move, scale etc), redraw
        
        t = Track.get_from_group entity
        unless t == nil

          # Only do stuff if position changed.
          new_controls = entity.get_attribute Track::ATTR_DICT, "controls"
          unless new_controls == t.controls
            t.controls = new_controls

            # Add affected tracks to drawing queue.
            changed = t.update_connections
            unless Observers.undo_redo?
              t.queue_drawing
              Track.drawing_queue_ends += changed
              UI.start_timer(0, false) { Track.draw_queued(true) }
            end

            # Update rolling stock's relation to this track.
            t.unlink_rstocks
            RStock.find_tracks_for_all_not_on_tracks
            
          end
        
        end
      
      elsif RStock.group_is_r_stock? entity
        #Rolling stock changed
        puts "Rolling stock modified (onElementModified, Undo/Redo: #{Observers.undo_redo?})."
        
        rs = RStock.get_from_group entity
        if rs
        
          #Update position for RStock object from new position of group
          rs.update_position_from_model
          
        end
        
      end
      
      Observers.enable

    end

  end#class

  class MyEntityObserver < Sketchup::EntityObserver

    def onEraseEntity(entity)
    
      return if Observers.disabled?
      Observers.disable
      
      puts "Purge objects linked to deleted geometry (onEraseEntity, Undo/Redo: #{Observers.undo_redo?})."

      #This method doesn't seem to be triggered for every deleted entity, just once every time stuff is deleted.
      #Also the entity argument cannot be used to find a track object since it's deleted...
      #Instead just purge all track and rolling stock objects with a deleted group

      objects_deleted = ((RStock.instances + Track.instances).select { |i| i.group.deleted? })
      unless objects_deleted.empty?
        model = Sketchup.active_model
        model.start_operation("Post entity erased updates", true, false, true) unless Observers.undo_redo?
        objects_deleted.each { |i| i.delete! }
        model.commit_operation unless Observers.undo_redo?
      end
      
      Observers.enable

    end

  end#class

  class MyInstanceObserver < Sketchup::InstanceObserver

    def onOpen(entity)
      #If entering a track or structure group,
      #warn user that changes might be lost unless it's a group plugin wont draw automatically to.
      
      return if Observers.disabled?
      Observers.disable
      
      if Track.group_is_track? entity
        #Track
        
        track = EneRailroad::Track.get_from_group entity
        unless track.disable_drawing
        
          msg = S.tr("Track groups are by default automatically drawn to (e.g. when it's moved or an other track connects to or disconnects from it). This destroys changes made manually to the group.") + "\n\n" + S.tr("Do you want to prevent this track from being automatically drawn to?")
          if UI.messagebox(msg, MB_YESNO) == IDYES
            track.disable_drawing = true
            UI.start_timer(0, false) do
              model = entity.model
              model.start_operation S.tr("Prevent Automatic Track Drawing"), true
              track.save_attributes
              model.commit_operation
            end
          end
        
        end
      
      elsif ad = entity.attribute_dictionary(Structure::ATTR_DICT)
        #Structure
        #Read and write directly to attributes
        
        unless ad["disable_drawing"]
                
          msg = S.tr("Structure groups are by default automatically drawn to (e.g. when it's moved or from structure properties). This destroys changes made manually to the group.") + "\n\n" + S.tr("Do you want to prevent this structure from being automatically drawn to?")
          if UI.messagebox(msg, MB_YESNO) == IDYES
            UI.start_timer(0, false) do
              entity.model.start_operation S.tr("Prevent Automatic Structure Drawing"), true
              ad["disable_drawing"] = true
              entity.model.commit_operation
            end
          end
        
        end
      
      end
      
      Observers.enable
      
    end

  end#class

  # Attach app observer if not already attached.
  @@app_observer ||= nil
  unless @@app_observer
    @@app_observer = MyAppObserver.new
    Sketchup.add_observer @@app_observer
  end

  # Initialize plugin for model when starting Sketchup and when reloading this
  # file.
  init_model Sketchup.active_model

end# Module

end# Module
