# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class RStock
  #A rolling stock is either a railroad car or locomotive.
  #Each rolling stock is a part of a train (other class), even when not connected to other rolling stocks.
  #The rolling stock is linked to a Sketchup::CompontntInstance in the model.

  #RStock object itself cannot be saved when the application closes but all data is saved as attributes to the component instance.
  
  # Public: Name of attribute dictionary for data in rolling stock group.
  ATTR_DICT = "#{ID}_r_stock"
  
  # Public: Name of attribute dictionary for data regarding moving parts
  # inside rolling stock.
  ATTR_DICT_PART = "#{ID}_mechanic"

  #All loaded rolling stock objects. In mac these might be in different models
  @@instances ||= []

  #Class variable accessors

  def self.instances; @@instances; end
  def self.instances=(value); @@instances = value; end

  #Class methods
   
  def self.get_from_group(group)
    #Returns rolling stock object for given group
    #Returns nil if group doesn't represent a track

    @@instances.find { |i| i.group == group }

  end

  def self.get_from_selection
    #Get object for selected rolling stock
    #Only one entity should be selected
    #Returns nil if not a rolling stock

    ss = Sketchup.active_model.selection
    return unless ss.length == 1

    self.get_from_group ss[0]
  
  end
  
  def self.group_is_r_stock?(entity)

    return false unless entity.is_a?(Sketchup::Group)
    return false unless entity.attribute_dictionary ATTR_DICT
    
    true
    
  end
  
  def self.find_tracks_for_all_not_on_tracks
    #Check all rolling stocks that don't have track references to see if they are in fact on tracks.
    #Runs after adding new tracks, moving tracks or undo/redo (since tracks may have changed).
    
    @@instances.each do |rs|
      next if rs.tracks.all?
      rs.find_tracks
    end
    
    true
  
  end
  
  #Instance attribute accessors

  attr_accessor :points#A rolling stock has 2 fix points that follows the track path. the 0th is the frontmost one in the current direction of travel.
  attr_accessor :distance_buffers
  attr_accessor :tracks#The tracks each fix point is located on. Used to calculate next position when moving.
  attr_accessor :tangents#In what direction bogies should point. point outwards from rolling stock center.
  attr_accessor :vectors_coupling#Tells in what direction couplings should point.
  attr_accessor :reversed#Tells if the group is visually reversed compared to the control points. This boolean changes when the direction of travel changes.
  attr_accessor :distance_traveled#Used for mechanic parts such as wheels
  attr_accessor :group

  attr_accessor :train_name#Name of the train rolling stock is a part of, used to initialize trains when model loads
  attr_accessor :train_position#Position in train, also used to initialize
  attr_accessor :train_v#Train speed & acceleration, saves in all rolling stocks in train so it can be retrieved when re-opening model
  attr_accessor :train_v_target
  attr_accessor :train_a
  attr_accessor :train_autopilot_mode

  #Instance methods

  def initialize(group)
    #Create rolling stock object.
    #Can be initialized from existing group

    #Load attributes from component instance
    group.attribute_dictionary(ATTR_DICT).each_pair do |key, value|
      self.instance_variable_set("@" + key, value)
    end

    #Set distance traveled to 0 if not already set.
    @distance_traveled ||= 0

    #Override group reference since group cannot be saved as attribute
    @group = group

    #Attach observer
    @group.add_observer(Observers::MyInstanceObserver.new)
    @group.add_observer(Observers::MyEntityObserver.new)

    #Add rolling stock to list
    @@instances << self

  end

  def delete!
    #Removes rolling stock from its train, deletes rolling stock from rolling stock array
    #Splits train if necessary

    train = self.train

    if train.r_stocks.length == 1
      #Remove train if rolling stock was alone in train
      train.delete!

    else
      #Remove rolling stock from train

      #Split train if rolling stock is not in its end
      #train should still refer to the train self is a part of
      #self should be last rolling stock of that train, meaning the train_position of all the others aren't affected
      if ![train.r_stocks[0], train.r_stocks[-1]].include? self
        train.split @train_position, false
      elsif train.r_stocks[0] == self
        #If first rolling stock of train is removed, lower train_position by one on all the others
        train.r_stocks.each { |rs| rs.train_position = rs.train_position - 1 }
      end

      #Remove rolling stock from train
      train.r_stocks.delete self
    end

    #Remove rs from rs instance array
    RStock.instances.delete self
    
    #Remove rs from follow by camera
    ani = Animate.get_from_model
    ani.followed = nil if ani && ani.followed == self

    true

  end

  def draw_to_position
    #Moves rolling stock to its current position according to @points
    #Used in train.move and inserting rolling stock.

    #Reverse controls when traveling backwards.
    #In class instance variables the first element is always the frontmost in train,
    #in local scope first element corresponds to the side closest to the group origin.
    points = @points.dup
    tangents = @tangents.dup
    vectors_coupling = @vectors_coupling.dup if @vectors_coupling
    if @reversed
      points.reverse!
      tangents.reverse!
      vectors_coupling.reverse! if vectors_coupling
    end

    #Transform group
    vector_along = points[0]- points[1]
    vector_perp = Z_AXIS.cross(vector_along)
    trans = Geom::Transformation.axes(points[0], vector_along, vector_perp, vector_along.cross(vector_perp))
    @group.move! trans
   
    #Move child entities
    
    #End specific parts (bogies and couplings)
    bogie_groups = []#reference to groups used as bogies for moving parts inside bogies. Only applies to groups since they must be unique.
    0.upto(1) do |end_index|
      @group.entities.each do |e|
        next unless [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class)

        #Rotate bogies
        if e.name == "bogie#{end_index}" && !tangents.empty?

          p = points[end_index].clone
          p.transform!(trans.inverse)
          tangent = tangents[end_index].clone
          tangent.transform!(trans.inverse)
          perp = Z_AXIS.cross(tangent)
          trans_b = Geom::Transformation.axes(p, tangent, perp, tangent.cross(perp))
          e.move! trans_b
          
          #Save group reference for moving parts inside group
          bogie_groups[end_index] = e if e.class == Sketchup::Group

        end

        #Rotate couplings
        if e.name == "coupling#{end_index}" && vectors_coupling && !vectors_coupling.empty?

          p = points[end_index].clone
          p.transform!(trans.inverse)
          vectors_couplin = vectors_coupling[end_index].clone
          vectors_couplin.transform!(trans.inverse)
          perp = Z_AXIS.cross(vectors_couplin)
          trans_c = Geom::Transformation.axes(p, vectors_couplin, perp, vectors_couplin.cross(perp))
          e.move! trans_c

        end
      end
    end

    #Tilt car body
    #Tilting part should be a group or component with its coordinates matching the rolling stock group when not tilting.
    #Origin should be on rotation axis.
    @group.entities.each do |e|
      next unless [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "tilting"
      
      #Get center point
      plane_start = [points[0], MyGeom.flatten_vector(@tangents[0])]
      plane_end = [points[1], MyGeom.flatten_vector(@tangents[1])]
      line_intersect = Geom.intersect_plane_plane plane_start, plane_end
      point_center =
      if line_intersect
        line_intersect[0]
      else
        Geom::Point3d.linear_combination(0.5, points[0], 0.5, points[1])
      end
      point_center.z = (points[0].z + points[1].z)

      trans_old = e.transformation
      roll_old = Math.acos trans_old.to_a[5]
      roll_old *= -1 if trans_old.to_a[6] < 0
      
      if line_intersect
        radius = (points[0].distance(point_center) + points[1].distance(point_center))*0.5
        radius = radius.to_m#radius to meters to match unit of v
        
        ag = 9.82
        ac = self.train.v**2/radius
        roll = Math.atan(ac/ag)
        
        #Make roll negative when turning one way
        right = (@points[0]-@points[1])*Z_AXIS#vector point right relative to default direction of travel
        point_righ = points[0].offset right
        if point_center.distance(point_righ) > point_center.distance(points[0])
          roll *= -1
        end
      else
        roll = 0
      end
      
      ##Smoothen roll
      time_since_last_frame = Animate.get_from_model.time_since_last_frame
      if time_since_last_frame
        smooth_cofficient = 0.95*time_since_last_frame
        roll = roll*smooth_cofficient + roll_old*(1-smooth_cofficient)
      end
      
      vector = Geom::Vector3d.new(1,0,0)
      point = trans_old.origin#Use current origin as rotation point
      trans_rotate = Geom::Transformation.rotation ORIGIN, vector, roll
      trans_translate = Geom::Transformation.translation(point - ORIGIN)
      trans = trans_translate*trans_rotate
        
      e.move! trans
      break#allow multiple entities to tilt?
    end
    
    #Move mechanic parts (mostly at steam locomotives)
    #These can be either in rolling stock root or inside bogies that are groups (not bogies that are components since that would make the parts move on all instances of the bogie)
    possibly_mechanic_entities = bogie_groups.map { |i| i.entities.to_a }.flatten
    possibly_mechanic_entities += @group.entities.to_a
    possibly_mechanic_entities.each do |e|
      next unless [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class)
      ad = e.attribute_dictionary ATTR_DICT_PART
      next unless ad
      
      #DO NOT GET GEOMETRIC DATA FROM SELECTION VERTICES, THEY ARE RELATIVE TO GLOBAL COORDINATES; NOT ROLLING STOCK GROUP COORDINATES.
      #Sketchup.active_model.selection[0].length is safe (since rolling stock group shouldn't be scaled)

      #Drawing specification
      # Located in rolling stock root or bogies that are groups (not components since components are unique)
      # Initialize AFTER initializing rolling stock.
      # Don't move or copy entities after initializing with data specific for a location (true for all but rotation), attributes won't be updated to match new location.
      # Instead initialize all parts but rotating separately.
      # To reset distance traveled, select rolling stock group and run "rs=EneRailroad:RStock.get_from_selection;rs.distance_traveled=0;rs.draw_to_position".

      #Attribute specification
      # "cycle" (required for all)
      #   The length the train travels until the cycle is back at its original state.
      #   Measured in inches, Sketchup's internal length unit.
      # "cycle_offset" (optional)
      #   How much "ahead" a specific group/component should be.
      #   Useful for coupling rods.
      #   Measured in cycles, values between 0 and 1. Usually 0.25 for one of the sides of a steam locomotive.

      cycle = ad["cycle"]
      cycle_offset = ad["cycle_offset"] || 0

      puts "No cycle set for mechanic part in rolling stock!\n  Rolling stock: #{@group.name}\n  Motion Type:   #{ad["motion_type"]}\n" unless cycle

      #Angle to base part's position on
      angle = (@distance_traveled/cycle%1+cycle_offset)*2*Math::PI
      
      #Compensate for difference in coordinates in last bogie.
      #(this is going "backwards" in a sense with x axis pointing the other direction.)
      if e.parent.instances[0] != @group && e.parent.instances[0].name == "bogie1"
        angle = -angle
      end
      
      trans_old = e.transformation

      case ad["motion_type"]
      when "rotation"
        #Rotation (e.g. wheel axis)

        #Group/component specification
        # Origin should be on the rotation axis.
        # Internal axes should match rolling stock group axes in initial state (when distance traveled (adjusted to cycle_offset) is 0)

        #Attribute specification
        # "rotation_vector" (optional)
        #   Can point in any direction.
        #   Along positive y means rotation matches direction of travel (wheels on top of the track) and is default value used when attribute isn't set.
        #   MUST be saved as Array, not Vector3d, to stop it from transforming when group moves.

        vector = Geom::Vector3d.new(ad["rotation_vector"] || [0, 1, 0])#use y axis if not defined
        point = trans_old.origin#Use current origin as rotation point
        trans_rotate = Geom::Transformation.rotation ORIGIN, vector, angle
        trans_translate = Geom::Transformation.translation(point - ORIGIN)
        trans = trans_translate*trans_rotate

      when "translation_circular"
        #Translation around axis
        #E.g. used on coupling rod.

        #Group/component specification
        # Rotates around rolling stock y axis unless rotation_vector is defined.
        # Keep axes matching rolling stock axes to match rotation of wheels (if they are on top of the track)
        # In top position (in internal z axis) for its initial state unless vector_to_start_position is set.

        #Attribute specification
        # "rotation_point"
        #   The point to rotate group/component origin around.
        #   MUST be saved as Array, not Point3d, to stop it from transforming when group moves.
        # "radius"
        #   Radius between rotation point and group/component origin.
        #   Measured in inches.#Attribute specification
        # "rotation_vector" (optional)
        #   Can point in any direction.
        #   Along positive y means rotation matches direction of travel (wheels on top of the track) and is default value used when attribute isn't set.
        #   MUST be saved as Array, not Vector3d, to stop it from transforming when group moves.
        # "vector_to_start_position" (optional)
        #   Initial direction from rotation_point to rotating point (internal origin), by default positive z in internal coordinates.

        point = Geom::Point3d.new ad["rotation_point"]
        radius = ad["radius"]
        vector = Geom::Vector3d.new(ad["rotation_vector"] || [0, 1, 0])#use y axis if not defined
        vector_to_start_position = ad["vector_to_start_position"] ? Geom::Vector3d.new(ad["vector_to_start_position"]) : trans_old.zaxis
        vector_to_start_position.length = radius
        trans_rotate = Geom::Transformation.rotation ORIGIN, vector, angle
        vector_to_start_position.transform! trans_rotate
        new_origin = point.offset vector_to_start_position
        trans = Geom::Transformation.axes new_origin, trans_old.xaxis, trans_old.yaxis, trans_old.zaxis

      when "translation_linear_harmonic"
        #Translation_linnear (harmonic oscillator)

        #Group/component specification
        # Goes back and forth along internal x axis

        #Attribute specification
        # "center_point"
        #   Point where group/component origin is in its initial state.
        #   MUST be saved as Array, not Point3d, to stop it from transforming when group moves.
        # "amplitude"
        #   Maximum distance in each direction along internal x axis.
        #   Measured in inches.

        point = ad["center_point"]
        amplitude = ad["amplitude"]
        along_vector = trans_old.xaxis
        position_length = amplitude*Math.sin(angle)
        along_vector.length = position_length
        new_origin = point.offset along_vector
        trans = Geom::Transformation.axes new_origin, trans_old.xaxis, trans_old.yaxis, trans_old.zaxis

      when "combined_rotation_reciprocation"
        #One end has rotational motion, one has a reciprocating motion.

        #Group/component specification
        # Origin should be on the rotation axis.
        # Internal x axis should point towards the end that moves along a line.

        #Attribute specification
        # "rotation_point"
        #   Point origin rotates around
        #   MUST be saved as Array, not Point3d, to stop it from transforming when group moves.
        # "rotation_vector" (optional)
        #   Vector defining rotation axis. y axis by default.
        # "radius"
        #   Radius between rotation point and group/component origin.
        #   Measured in inches.
        # "length"
        #   Length from origin along internal x axis to point moving along a line.
        #   Measured in inches.
        # "line"
        #   The line the 'other' (not origin) end of the rod follows.
        #   Saved as an array containing the 2 elements point and vector.
        #   point and vector MUST be saved as Array, not Point/Vector3d, to stop it from transforming when group moves.
        # "vector_to_start_position" (optional)
        #   Initial direction from rotation_point to rotating point (internal origin), by default upwards in rolling stock group coordinates.

        point = Geom::Point3d.new ad["rotation_point"]
        radius = ad["radius"]
        length = ad["length"]#length (along internal x axis) to the reciprocation point.
        line = ad["line"]#line that point moves along. should be on same internal x y plane as point.
        line = [Geom::Point3d.new(line[0]), Geom::Vector3d.new(line[1])]
        vector_rotation = Geom::Vector3d.new(ad["rotation_vector"] || [0, 1, 0])#use y axis if not defined
        vector_to_start_position = Geom::Vector3d.new(ad["vector_to_start_position"] || [0, 0, 1])#Vector to start position is by default positive z axis in rolling stock

        vector_to_start_position.length = radius
        trans_rotate = Geom::Transformation.rotation ORIGIN, vector_rotation, angle
        vector_to_start_position.transform! trans_rotate
        point_rotating = point.offset vector_to_start_position

        intersections = MyGeom.intersect_line_sphere line, point_rotating, length
        
        if intersections.empty?
          puts "No intersections found in combined_rotation_reciprocation! perhaps length is to short."
          puts "  length:          " + length.inspect
          puts "  point_rotating:  " + point_rotating.inspect
          puts "  line:            " + line.inspect
          puts
          next
        end
        
        other_point = #intersection closest to point defining line
        if intersections[0].distance(line[0]) < intersections[1].distance(line[0])
          intersections[0]
        else
          intersections[1]
        end
        new_x = other_point - point_rotating

        trans = Geom::Transformation.axes point_rotating, new_x, trans_old.yaxis, new_x*trans_old.yaxis

      when "translation_linear_reciprocation"
        #Reciprocating translation motion.

        #Group/component specification
        # Origin corresponds to the point moving linear in combined_rotation_reciprocation.

        #Attribute specification
        # <All on corresponding combined_rotation_reciprocation> + motion_type

        point = Geom::Point3d.new ad["rotation_point"]
        radius = ad["radius"]
        length = ad["length"]#length (rod's along internal x axis) to the reciprocation point.
        line = ad["line"]#line that point moves along.
        line = [Geom::Point3d.new(line[0]), Geom::Vector3d.new(line[1])]
        vector_rotation = Geom::Vector3d.new(ad["rotation_vector"] || [0, 1, 0])#use y axis if not defined
        vector_to_start_position = Geom::Vector3d.new(ad["vector_to_start_position"] || [0, 0, 1])#Vector to start position is by default positive z axis in rolling stock

        vector_to_start_position.length = radius
        trans_rotate = Geom::Transformation.rotation ORIGIN, vector_rotation, angle
        vector_to_start_position.transform! trans_rotate
        point_rotating = point.offset vector_to_start_position

        intersections = MyGeom.intersect_line_sphere line, point_rotating, length
        
        if intersections.empty?
          puts "no intersections found in translation_linear_reciprocation. perhaps length is to short."
          puts "length:          " + length.inspect
          puts "point_rotating:  " + point_rotating.inspect
          puts "line:            " + line.inspect
          puts
          next
        end
        
        other_point = #intersection closest to point defining line
        if intersections[0].distance(line[0]) < intersections[1].distance(line[0])
          intersections[0]
        else
          intersections[1]
        end

        trans = Geom::Transformation.axes other_point, trans_old.xaxis, trans_old.yaxis, trans_old.zaxis

      end#case
      
      e.move! trans

    end#each

  end

  def draw_text(texts, wrap_in_operation = true)
    #Rewrites 3d text in rolling stock group.
    #text is an array containing hashes of what to draw.
    #Each hash contains the text itself and a label.

    #Example: rolling_stock.draw_text([{:label => "Number", :text => "007"}])

    Sketchup.active_model.start_operation(S.tr("Change Rolling Stock Text"), true) if wrap_in_operation

    #Find text groups
    tilting_body = @group.entities.find { |e| [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "tilting" }#(only one tilting body allowed per rolling stock)
    ents = @group.entities.to_a
    ents += tilting_body.entities.to_a if tilting_body
    ents.each do |group|
      next unless group.class == Sketchup::Group
      next unless group.name
      regex_results = /\Atext_(.*)\Z/i.match(group.name)
      next unless regex_results

      label = regex_results[1]
      text_hash = texts.find { |i| i[:label].downcase == label.downcase }

      if text_hash

        font = group.get_attribute ATTR_DICT_TEXT, "font"#Font cannot be manually set but can be added to the group by ruby command
        EneRailroad.write_text group, text_hash[:text], font
        group.set_attribute ATTR_DICT_TEXT, "string", text_hash[:text]

      end

    end

    Sketchup.active_model.commit_operation if wrap_in_operation

  end

  def find_tracks
    #Find what tracks rolling stock is placed on (one reference for each anchor point).
    #Runs when initializing on model load, when inserting rolling stock and when rolling stock is manually moved (observer).

    @tracks = []
    @tangents = []
    
    return if Track.instances.empty?
    
    calibration_vector = @points[1] - @points[0]#vector in rolling stock's direction
    0.upto(1) do |rs_end|
      q =  Track.inspect_point(@points[rs_end], model)
      if q[:distance] > 0
        tracks << nil
      else
        tracks << q[:track]
      end
      
      vector = q[:vector]
      vector.reverse! if calibration_vector.angle_between(vector) < 90.degrees#make sure tangent is not 180 degrees wrong
      tangents << vector
      calibration_vector.reverse!#tangents should point away from each other, change calibration vector
    end

    true

  end

  def inspect
    #Get inspect string consistent to Sketchup's own classes without relying on altered to_s
    "#<#{self.class}:0x#{(self.object_id << 1).to_s(16)}>"
  end

  def list_texts
    #List texts and labels in rolling stock.
    #Return array.
    #Element looks like {:label => "Destination", :text => "Knuffingen"}.

    texts = []

    #Find text groups
    tilting_body = @group.entities.find { |e| [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "tilting" }
    ents = @group.entities.to_a
    ents += tilting_body.entities.to_a if tilting_body
    ents.each do |group|
      next unless group.class == Sketchup::Group
      next unless group.name
      regex_results = /\Atext_(.*)\Z/i.match(group.name)
      next unless regex_results

      label = regex_results[1]
      text = group.get_attribute("#{ID}_text", "string") || ""

      texts << {:label => label, :text => text} if texts.select { |i| i[:label] == label} .empty?

    end#each

    texts

  end

  def length_over_couplings
    #Returns the length from one buffer to the other
    
    @distance_buffers[0] + @distance_buffers[1] + (@points[0] - @points[1]).length
    
  end
  
  # Public: Model rolling stock is drawn to or active Model if rolling stack has
  # not been drawn.
  #
  # Returns a Model.
  def model
  
    @group && @group.valid? ? @group.model : Sketchup.active_model
    
  end
  
  def reverse_controls
    #Switches start and end control points and distance_buffers.
    #Used when reversing train.
    #Does not affect model visually.

    @distance_buffers.reverse!
    @points.reverse!
    @tracks.reverse!
    @tangents.reverse!
    @vectors_coupling.reverse! if @vectors_coupling

    #Keeps track on controls being reversed compared to group
    #Used in draw_to_position to keep model orientation when controls are reversed
    @reversed = !@reversed

    true

  end

  def text_UI
    #Open UI to Set text on signs on the rolling stock.

    #Find text groups in rolling stock along with their text and label
    #This is done before the inputbox so user is only asked to change texts that exists
    texts = self.list_texts

    if texts.empty?
      UI.messagebox(S.tr("There are no texts in this rolling stock.") + "\n" + S.tr("'Plugins > Railroad > Documentation > Rolling Stock' describes how to add texts to rolling stocks."))
      return
    end#if

    #Ask for new text
    labels = texts.map { |t| t[:label] }
    defaults = texts.map { |t| t[:text] }
    input = UI.inputbox(labels, defaults, S.tr("Change Rolling Stock Text"))
    return unless input
    input.each_with_index do |v, i|
      texts[i][:text] = v
    end

    #Draw new text to rolling stock
    self.draw_text texts

    true

  end

  def save_attributes
    #Writes instance variables to group attributes so they are saved when application closes.
    #Called before save.

    #Also save some train variables that aren't saved to rolling stock each time they are changed (train.v changes in each frame when accelerating for instance)
    @train_v = self.train.v
    @train_v_target = self.train.v_target
    @train_a = self.train.a
    @train_autopilot_mode = self.train.autopilot_mode

    self.instance_variables.each do |i|
      key = i
      value = self.instance_variable_get(key)
      key = key.to_s  #Make string of symbol
      key[0] = ""    #Remove @ sign from key
      next if key == "group"
      @group.set_attribute ATTR_DICT, key, value
    end#each

  end

  def to_s
    @group.name
  end

  def train
    #Returns the train rolling stock is a part of

    Train.instances.find { |t| t.r_stocks.include? self }

  end

  def update_position_from_model
    #Updates points and track references from group position in model.
    #Called from onElementModified observer.
    
    #Can't use attributes since they don't seem to change when group.move! is called
    #Use group transformation instead
    trans = @group.transformation
    points = [trans.origin]
    along = trans.xaxis.reverse
    wheelbase = @points[0].distance(@points[1])
    along.length = wheelbase
    points[1] = points[0].offset along
    points.reverse! if @reversed
    @points = points
    
    #Find tracks rolling stock stands on
    self.find_tracks
        
  end
  
  def uninit!
    #Un-initialize rolling stock group.
    #This can be done if it was initialized (in insert rolling stock tool) with faulty parameters.
    
    @group.attribute_dictionaries.delete ATTR_DICT
    self.delete!
    
  end
  
end#class

end#module
