# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolRStockInsert
  #This tool is used to add rolling stocks,
  #either predefined or creating own from group.

  #The predefined rolling stock from the rolling stock directory may actually
  #be several rolling stocks that are all added at once, such as a steam locomotive
  #along with it's tender.

  def initialize

    #What is user doing?
    #Create is when a group is turned into a rolling stock.
    #Place is when a rolling stock is placed on the tracks,
    #either from library, selection or newly created from group
    #@mode changed further down in initialize if a suitable group for creating rolling stock is selected
    @mode = "place"


    #Place mode

    #Id of rolling stock (collection) placed
    #Corresponds to a rolling stock file that can contain multiple rolling stocks (e.g. steam locomotive + tender)
    #Defined in load_rs
    @rs_id
    
    #Rolling stock hovered with mouse. used to attach rolling stocks to existing train. rolling stock object or nil
    @rs_hovered
    
    #End of the train hovered. 0 (front) or 1 (end)
    @train_hovered_end
    
    #Track input point is on (or hovered rolling stock is on). used to calculate point to place rolling stocks on
    @track

    #What end of the loaded rolling stock(s) is being placed. 0 for front, 1 for end.
    @placement_end = 0

    #Distances between @points_place, sets when in rs_load
    #Order corresponds to order in rolling stock model file and is independent from @placement_end
    @distances = []

    #Point along track (only required as class instance var to be drawn, otherwise only used within mousemove)
    @point

    #Array of points where to place the wheels/axis
    #First and last point is where the first and last coupling will be
    #Order corresponds to model file and is indepenent from placement_end
    #Created in onMouseMove
    #False means that user didn't hover track or train, nil means track ended, array means all point could fit
    @points_place = false

    #Vector is approximate and tells in which direction along track to place train
    #Only used when placed on track. when placed towards existing train the end of it is used
    #a little diagonal to give less fluctuating results on tracks that otherwise would be right angled to vector
    @vector = Geom::Vector3d.new(10, 1, 0)

    #Only affects look of tool, not drawn results
    @default_vector_length = 9.m


    #Create mode

    #Tool stage telling on what step in the creation process the user is
    @stage = 0
    #The results of each stage, such as a point or an edge
    @stage_results = []


    #Change mode to create if a suitable group is selected
    ss = Sketchup.active_model.selection
    if ss.length == 1 && self.is_valid_group_for_rs?(ss[0])
      @mode = "create"
      @stage = 1#Group is already selected
      @stage_results << ss[0]
      ss.clear
    end

    #Input point
    @ip = nil

    #Statusbar texts
    @status_text_place = S.tr "Click on track or train to insert rolling stock(s). Tab = Toggle end to place, Alt = Opposite direction."
    @status_text_create = S.tr "Follow instructions in dialog to create rolling stock."
    @status_text = @mode == "place" ? @status_text_place : @status_text_create

    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "r_stock_insert.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def activate

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    #Prevents select tool from being selected every time this tool is deactivated. This should only happen when dialog is manually closed
    @tool_active = true

    Sketchup.set_status_text(@status_text, SB_PROMPT)

    #Open web dialog with tool settings
    self.init_web_dialog

  end

  def deactivate(view)

    @tool_active = false
    @dialog.close
    view.invalidate

  end

  def draw(view)

    color = Sketchup::Color.new("Magenta")
    color2 = Sketchup::Color.new("White")

    if @mode == "place"
      #A rolling stock is being placed
      #Show a solid line for each wheelbase and a dotted line between each rolling stock
      #Show an arrow in the middle of each wheelbase telling direction of rolling stock

      if @points_place
        #Points exists (fits to railroad)

        #Draw point at wheels/axes
        view.draw_points @points_place, 7, 4, color2

        #Draw lines for wheelbases and couplings
        view.line_width = 1
        view.drawing_color = color2
        #every second line is a coupling (dotted) and every second is a wheelbase (solid)
        0.upto(@points_place.length-2) do |i|
          point = @points_place[i]
          point_next = @points_place[i+1]
          if i%2 > 0
            #Within rolling stock
            view.line_stipple = ""
            MyView.draw_arrow(view, Geom.linear_combination(0.5, point, 0.5, point_next), point-point_next, 1.m, true)
          else
            #Between rolling stocks (couplings)
            view.line_stipple = "_"
          end
          view.draw_line point, point_next
        end

      #Draw input point
      @ip.draw view

      end

      unless @points_place == false
        #Track or rolling stock is hovered = points either exists or couldn't be calculated because track was to short, draw vector and point

        @vector.length = @default_vector_length

        #Point where track is being placed
        view.draw_points [@point], 10, 2, color

        #Vector line
        view.line_width = 3
        view.line_stipple = ""
        view.drawing_color = color
        view.draw_line @point, @point.offset(@vector)

      end#unless

      if @points_place == nil
        #Points could not be calculated, railroad ended before last point

        #Draw input point
        @ip.draw view

        view.tooltip = S.tr "Reached track end."#NOTE: SU ISSUE: red text when possible
        return#Without return tooltip isn't written, even though no more code is executed
        
      elsif @points_place == false
        #Not hovering rolling stock or track

        view.tooltip = S.tr "Must be placed on track"#NOTE: SU ISSUE: red text when possible
        return#Without return tooltip isn't written, even though no more code is executed
        
      end

    else
      #A rolling stock is created

      case @stage
      when 1
        #Select a point on top of rail plane
        MyView::draw_plane_outline view, [@ip.position, Z_AXIS], 1.m
      when 2
        #Select direction of travel edge
        edge = @ip.edge
        if edge
          points = edge.vertices.map { |v| v.position.transform(@ip.transformation) }
          view.drawing_color = view.model.rendering_options["HighlightColor"]
          view.line_width = 3
          view.draw_line points
        end
      when 3
        #Select point to define symmetry plane
        MyView.draw_plane_outline view, [@ip.position, Z_AXIS*@stage_results[2]], 1.m
      when 4..7
        #Axis/Bogie and buffer planes
        MyView.draw_plane_outline view, [@ip.position, @stage_results[2]], 1.m
      end

      #Draw input point
      @ip.draw view

    end

  end

  def onReturn(view)

    #Place rolling stock
    self.place_rs if @mode == "place"

  end

  def onMouseMove(flags, x, y, view)

    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y

    if @mode == "place"
      #A rolling stock is being placed

      #Assume nothing relevant was hovered
      @point = nil
      @rs_hovered = nil
      @track = nil

      #Determine how to place rolling stocks based on mouse position
      if picked
        if Track.group_is_track? picked
          #Hovering track

          @point = Track.inspect_point(@ip.position, view.model)[:point]
          @track = Track.get_from_group picked

        elsif picked.attribute_dictionary RStock::ATTR_DICT
          #Hovering train

          train = RStock.get_from_group(picked).train
          @train_hovered_end = (train.points[0].distance(@ip.position) < train.points[1].distance(@ip.position)) ? 0 : 1 #0 for front, 1 for end

          @point = train.points[@train_hovered_end]
          @rs_hovered = train.r_stocks[-@train_hovered_end]
          @vector = @rs_hovered.points[0] - @rs_hovered.points[1]
          @vector.reverse! if @train_hovered_end == 1
          @track = train.tracks[@train_hovered_end]

        end
      end

      #Calculate the points rolling stocks should be placed at
      self.calc_points

    else
      #A rolling stock is created

      case @stage
      when 0
        #Select group
        ss = Sketchup.active_model.selection
        ss.clear
        if self.is_valid_group_for_rs? picked
          #group is valid
          ss.add picked
        end
      end

    end

    view.invalidate

  end

  def onLButtonDown(flags, x, y, view)

  @ph.do_pick(x, y)
  picked = @ph.best_picked
  @ip.pick view, x, y

  if @mode == "place"
      #A rolling stock is being placed

      self.place_rs

   else
      #A rolling stock is being created

      case @stage
      when 0
        #Select group
        if self.is_valid_group_for_rs? picked
          #group is valid
          Sketchup.active_model.selection.clear
          @stage_results << picked
          @stage += 1
        end
      when 1
        #Select a point on top of rail plane
        @stage_results << @ip.position
        @stage += 1
      when 2
        #Select direction of travel edge
        edge = @ip.edge
        if edge
          #Save the points and not the edge itself
          points = edge.vertices.map { |v| v.position.transform(@ip.transformation) }
          vector = points[1] - points[0]
          if vector.parallel? Z_AXIS
            UI.messagebox S.tr("Direction of travel cannot be vertical.")
            return
          end
          @stage_results << vector
          @stage += 1
        end
      when 3
        #Select point to define symmetry plane
        @stage_results << @ip.position
        @stage += 1
      when 4
        #First axis/bogie
        @stage_results << @ip.position
        @stage += 1
      when 5
        #second axis/bogie
        
        #Check so this point isn't on same plane as previous one, wheelbase can't be 0
        line = [@stage_results[4], @stage_results[2]]
        input_projected = @ip.position.project_to_line line
        return if input_projected == @stage_results[4]
        
        @stage_results << @ip.position
        @stage += 1
      when 6
        #first buffer
        @stage_results << @ip.position
        @stage += 1
      when 7
        #second buffer and last stage
        @stage_results[7] = @ip.position#uses index instead of adding to array since this stays as the current stage if inputbox is canceled

        #Input box
        prompts = [S.tr("Rolling Stock Name")]
        keys = [:rs_name]
        defaults = [@stage_results[0].name]#group.name
        input = UI.inputbox(prompts, defaults, S.tr("Create Rolling Stock"))
        return unless input
        input = Hash[*keys.zip(input).flatten]

        #Find points and buffer distances
        group = @stage_results[0]
        tor_point = @stage_results[1]
        direction = @stage_results[2]
        sideways = Z_AXIS*direction
        rs_up = direction*sideways#rs might be created on sloping rails
        rail_plane = [tor_point, rs_up]
        symmetry_plane = [@stage_results[3], sideways]
        base_line = Geom.intersect_plane_plane rail_plane, symmetry_plane
        points = @stage_results[4..5].map { |p| p.project_to_line base_line }
        point_buffer = @stage_results[6..7].map { |p| p.project_to_line base_line }
        point_buffer.reverse! if point_buffer[0].distance(points[0]) > point_buffer[0].distance(points[1])#Make the zeroth buffer be the one closest to the zeroth point
        distance_buffers = [0, 1].map { |index| points[index].distance point_buffer[index] }

        #Create rolling stock
        Sketchup.active_model.start_operation "Create Rolling Stock", true

        #Group must be unique before it's edited.
        group.name += ""#Group.make_unique is deprecated and this seems to work instead

        #Change group origin and axis
        newX = points[0] - points[1]
        newZ = Z_AXIS
        newY = newZ.cross newX
        trans_new = Geom::Transformation.axes points[0], newX, newY, newZ
        MyGeom.move_group_origin group, trans_new

        #Set attributes. The rolling stock initializer uses these values
        name = Train.unique_name(input[:rs_name])
        group.set_attribute RStock::ATTR_DICT, "points", points
        group.set_attribute RStock::ATTR_DICT, "distance_buffers", distance_buffers
        group.set_attribute RStock::ATTR_DICT, "reversed", false
        group.set_attribute RStock::ATTR_DICT, "train_name", name
        group.set_attribute RStock::ATTR_DICT, "train_position", 0

        #Initialize rolling stock and train from group (same code as when loading a model)
        rs = RStock.new group
        rs.find_tracks
        Train.new [rs]

        Sketchup.active_model.commit_operation

        #reset stage
        @stage = 0
        @stage_results = []
        
        #Ask if rolling stock should be saved to library
        if UI.messagebox(S.tr("Do you want to save the rolling stock to library?") + "\n\n" + S.tr("You can also save it to library later from the context menu and then also save multiple rolling stocks at once, such as a steam locomotive along with its tender"), MB_YESNO) == 6
          Template.save_selected_r_stocks(rs) do |rs_id|
            #Update rolling stock list in library when rolling stock has been saved
            r_stock_json = Template.list_installed :r_stock, true
            js = "var r_stocks=#{r_stock_json};"
            js << "change_tab(0,0);"#Switch to library tab
            js << "update_library();"
            js << "select_rs('#{rs_id}');"#Select newly added rs
            @dialog.execute_script js
            @mode = "place"
          end
         end

      end

      #Update stage list and explanation in webdialog
      @dialog.execute_script "set_current_step(#{@stage});"

    end

    view.invalidate

  end

  def onKeyDown(key, repeat, flags, view)
    case key
    when 9#tab
      if @mode == "place"
        #Change rolling stock end being placed
        @placement_end = 1 - @placement_end
      end
    when ALT_MODIFIER_KEY
      if @mode == "place"
        #Reverse approximate train direction (will be adjusted to track)
        @vector.reverse!
      end
    end

    #Calculate new points where to place rolling stocks and redraw
    self.calc_points
    view.invalidate
    
    #Return true after alt is pressed to prevent focus from moving to File menu
    return true if key == ALT_MODIFIER_KEY && @mode == "place"

  end

  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate
  end

  def onCancel(reason, view)

    if @mode == "create"
      #A rolling stock is being created
      if @stage > 0
        @stage_results.pop
        @stage_results.pop if @stage == 7#Solving off-by-one error when canceling last stage
        @stage-= 1
        @dialog.execute_script "set_current_step(#{@stage});"
        view.invalidate
      end
    end

  end

  #Own definitions
  #Not called from Sketchup itself

  def init_web_dialog
    #Open web dialog with tool settings

    @dialog = UI::WebDialog.new(S.tr("Add Rolling Stock"), false, "#{ID}_insert_r_stock", 440, 440, 500, 100, true)
    @dialog.min_width = 440
    @dialog.min_height = 440
    @dialog.navigation_buttons_enabled = false
    @dialog.set_background_color @dialog.get_default_dialog_color
    @dialog.set_file(File.join(DIALOG_DIR, "insert_r_stock", "index.html"))

    #Create list of rolling stocks
    r_stock_json = Template.list_installed :r_stock, true
    js = "var r_stocks=#{r_stock_json};"

    #Translate strings
    js << S.tr_dlg
    
    #Retrieve what rolling stock library sorts and groups by
    js << "var sorting='#{Sketchup.read_default(ID, "r_stock_insert_tool_sorting") || "0"}';"
    js << "var grouping='#{Sketchup.read_default(ID, "r_stock_insert_tool_grouping") || "type"}';"
    
    js << "document.onkeyup=port_key;"
    js << "select_sorting_and_grouping();"
    js << "init_tabs();"
    
    #First tab corresponds to place mode, second tab is create mode
    js << "change_tab(0,#{@mode == "place" ? 0 : 1});"
    
    js << "update_library();"
    
    #Select first rolling stock in list (this also calls the callback for selecting rolling stock)
    #Comment out next line to not select any rs on start
    js << "select_rs(document.getElementById('library').getElementsByTagName('img')[0].getAttribute('data-id'));" unless r_stock_json == "[]"
    js << "set_current_step(#{@stage});"
    
    #Show dialog
    if WIN
      @dialog.show { @dialog.execute_script js }
    else
      @dialog.show_modal { @dialog.execute_script js }
    end
    
    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox
    
    #Change between place and create mode. changing mode will reset the creation process
    @dialog.add_action_callback("change_mode") { |_, callbacks|
      self.change_mode callbacks
    }

    #Select rolling stock in library
    @dialog.add_action_callback("select_rs") { |_, callbacks|
      load_rs(callbacks)
    }

    #Deselct tool when webdialog closes
    @dialog.set_on_close {
      Sketchup.send_action "selectSelectionTool:" if @tool_active
    }

    #Button opening rolling stock folder
    @dialog.add_action_callback("open_r_stock_folder") {
      Template.open_dir :r_stock
    }
    
    #Save variable on computer
    @dialog.add_action_callback("save_var") { |_, callbacks|
      a = callbacks.split "+"
      Sketchup.write_default ID, "r_stock_insert_tool_" + a[0], a[1]
    }

    #Open link in default browser
    @dialog.add_action_callback("open_in_default_browser") { |_, callbacks|
      UI.openURL callbacks
    }
    
    #Port key event to tool
    @dialog.add_action_callback("port_key") { |_, callbacks|
      self.onKeyDown callbacks.to_i, false, 0, Sketchup.active_model.active_view#This tool doesn't use flags so it can just be set to 0
    }

  end

  def is_valid_group_for_rs?(group)
    #Check if an entity can be used to create a rolling stock.
    #It has to be a group that isn't a track, structure or already a rolling stock.

    return unless group.class == Sketchup::Group
    return if Track.group_is_track? group
    return if Structure.group_is_structure? group
    return if group.attribute_dictionary(RStock::ATTR_DICT)

    true

  end

  def change_mode(new_mode)

    #Change between place and create mode
    @mode = new_mode
    @status_text = @mode == "place" ? @status_text_place : @status_text_create
    Sketchup.set_status_text(@status_text, SB_PROMPT)

    #Reset creation process, user can restart the creation process by clicking the create tab
    @stage = 0
    @stage_results = []

    #Toggle content in web dialog
    js = "change_tab(0,#{@mode == "place" ? 0 : 1});"
    js << "set_current_step(#{@stage});"
    @dialog.execute_script  js

  end

  def load_rs(id)
    #Loads rolling stock data.
    #Component definition is only loaded once the rolling stock(s) is placed
    #One file may contain several rolling stocks (such as steam locomotive + tender)
    #Called when rolling stock is selected from library

    @rs_id = id

    info = Template.info :r_stock, @rs_id
    placement_data = info[:placement_data]
    #placement_data uses Float for length since it can be serialized and saved but it doesn't seem to be a problem with that
    
    @distances = []
    #First coupling length has to be initialized before loop since each iteration adds to previous one.
    @distances[0] = 0
    placement_data.each do |rs1|
      @distances[-1] += rs1[:distance_buffers][0]
      @distances << rs1[:length_wheelbase]
      @distances << rs1[:distance_buffers][1]
    end#each
    
    #In calc_points the buffer distance of hovered rolling stock is added to end element of array

  end

  def calc_points
    #Calculates @points_place from @distances, @point, @placement_end and buffer distance of hovered_rs
    #Called from mousemove and pressing some keys
  
    if @point && !@distances.empty?
      #Find points where wheel/axis should be placed
      #First and last point in array is the end of the first/last coupling
      
      #Don't calculate points if hovering rs not on track
      if @rs_hovered && !@rs_hovered.tracks[@train_hovered_end]
        @points_place = nil
        return
      end

      #User local variable for distances since the array is reversed and values changed within this method
      distances = @distances.dup
      
      #Reverse distances array if the end of the rolling stock(s) is being placed instead of the front
      distances.reverse! if @placement_end == 1
      
      #If rolling stocks are added to existing train, add its buffer distance
      distances[0] += @rs_hovered.distance_buffers[@train_hovered_end] if @rs_hovered
      
      #Get points along track
      calc_along_results = Track.calc_points_along(@point, distances, @vector, @track)
      calc_along_results.map! { |i| i[:point] } if calc_along_results
      @points_place = calc_along_results
      
      #first point should be where the mouse cursor is projected to the track
      @points_place.unshift @point if @points_place
      
      #Reverse back points so the first point is the start of the rolling stocks, even when it's away from the input point
      @points_place.reverse! if @placement_end == 1 && @points_place
      
      #Update the vector determine rolling stock direction, this makes the tool consistent to insert track that "remembers" the vector of the last hovered track
      if @points_place
        if @placement_end == 0
          #Front being placed
          
          @vector = @points_place[1] - @points_place[0]
          #if buffer distance is 0, get vector from wheelbase instead
          @vector = @points_place[2] - @points_place[1] unless @vector.valid?
          
        else
          #Back end being placed
        
          @vector = @points_place[-2] - @points_place[-1]
          #if buffer distance is 0, get vector from wheelbase instead
          @vector = @points_place[-3] - @points_place[-2] unless @vector.valid?
          
        end
      end

    else
      @points_place = false#false means that user didn't hover track or train, nil means track ended, array means all point could fit
    end#if
      
  end
  
  def place_rs
    #Load component definition from sketchup file and place rolling stock(s)
    #according to @points_place
    #Called from mouse click or enter being pressed.
    
    #Rolling stocks should be copied into their own file named model.skp in the rolling stock directory.
    #There can be multiple rolling stocks in the same file, such as a steam locomotive and its tender.
    #Rolling stocks should be in the model root and have the normal attributes for points, distance_buffers etc.
    #train_position is used to determine their order.
    #train_name is unused and will be overwritten from the name in the info file.
    #point[0] must be at the first axis/boogie and point[1] at the back for the arrows indicating direction to work

    return unless @points_place
    
    #Load info file
    info = Template.info :r_stock, @rs_id
    
    #Get unique name for new train, train_name in rolling stock group attributes does not matter
    name = Train.unique_name info[:name]

    #Load model file
    component_definition = Template.component_def :r_stock, @rs_id
    rs_groups = component_definition.entities.select { |e| e.attribute_dictionary(RStock::ATTR_DICT) }
    rs_groups = rs_groups.sort_by { |g| g.get_attribute RStock::ATTR_DICT, "train_position"}
    
    Sketchup.active_model.start_operation(S.tr("Insert Rolling Stock"), true)
    
    #Prevent observer from updating points from group transformations.
    #This seems to create faulty position data if done.
    Observers.disable
    
    r_stocks_new = []

    #Loop rolling stocks to create (=every second point, exclude first and last)
    0.step(@points_place.length-3, 2) do |i|
    
      rs_index = i/2
      rs_group = rs_groups[rs_index]
      p0 = @points_place[i+1]
      p1 = @points_place[i+2]
      
      #Create transformation
      #Transformation is wrong for reversed rolling stocks but could as well be left out since it's overridden by draw_to_position later.
      vector_along = p0 - p1
      vector_perp = Z_AXIS * vector_along
      trans = Geom::Transformation.axes(p0, vector_along, vector_perp, vector_along.cross(vector_perp))
      
      #Place rolling stock group
      rs_group_definition = rs_group.entities.parent
      placed_group = Sketchup.active_model.entities.add_instance rs_group_definition, trans
      rs_group.attribute_dictionary(RStock::ATTR_DICT).each_pair do |key, value|
        placed_group.set_attribute RStock::ATTR_DICT, key, value
      end
      placed_group.set_attribute RStock::ATTR_DICT, "points", [p0,p1]
      placed_group.material = rs_group.material
      placed_group.name = name#Group.make_unique is deprecated but changing name seems to work too
      
      #Make bogies and tilt group unique so text and moving parts doesn't get mixed up between instances.
      make_uniq = placed_group.entities.select { |e| e.class == Sketchup::Group && e.name && ["bogie0", "bogie1", "tilting"].include?(e.name) }
      make_uniq.each { |g| g.name += "" }#Group.make_unique is deprecated but changing name seems to work too
      
      #Initialize r_stock
      rs = RStock.new placed_group
      rs.find_tracks
      #rs.draw_to_position#Done after train is initialized
      r_stocks_new << rs
      
    end
    
    #Create train with new rolling stock(s)
    train = Train.new r_stocks_new
    
    #Name train
    train.name = name
    
    #Position bogies and other moving parts inside rollings stocks.
    #This is done after initializing trains since tilting relies on train speed.
    #NOTE: couplings are misplaced in draw_to_position since find_tracks doesn't update coupling_vectors. perhaps coupling vectors should be taken directly from train in draw_to_position and not be attributes? then this method cannot be called until train has been initialized.
    train.r_stocks.each { |rs| rs.draw_to_position }
    
    Sketchup.active_model.commit_operation
    
    Observers.enable
    
    if @rs_hovered
      #Connect rolling stock to clicked train if not placed directly on tracks
      
      train_existing = @rs_hovered.train
      train_existing.join train, @train_hovered_end, @placement_end
    else
      #Update lights (when connecting to existing train this is done by the join method)
      
      train.draw_lights
    end
    
  end

end#class

end#module
