# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class Structure#NOTE: FEATURE: add text similar to rolling stock text.
  #Structures are elements drawn alongside tracks, e.g. tunnels, bridges, platforms or catenary.
  #The object is closely linked to a Sketchup::Group in the model root that the structure is drawn in.

  #Structure objects themselves cannot be saved when model is closed but their data is saved to their groups.
  #When needed the structure object is recreated from group attributes.
    
  # Arrayed parts of structures, e.g. pillars, can be assigned a serial_number
  # specific to its serial_name when drawn. These parts can then be listed by
  # their name and number.
  #
  # To add a serial_name to a certain part add the attribute serial_name to the
  # attribute directory ene_railroad_template_part of its Group or
  # ComponentInstance in the template model.
  
  # Public: Name of attribute dictionary for data in structure group.
  ATTR_DICT = "#{ID}_structure"
  
  @@export_path ||= ENV["HOMEPATH"] || ""
  
  #Class methods

  def self.get_from_group(group)
    #Returns structure object for given group
    #Returns nil if group doesn't represent a track

    #Structures objects are initialized only when needed.
    return unless group
    return unless group.attribute_dictionary ATTR_DICT
    
    self.new group

  end

  def self.get_from_selection
  #Get object for selected rolling structure
  #Only one entity should be selected
  #Returns nil if not a track
  
  #Since structure objects unlike other objects aren't memorized when not used a new temporary object will be created.

  ss = Sketchup.active_model.selection
  return unless ss.length == 1
  
  self.get_from_group ss[0]
  
  end
    
  def self.group_is_structure?(entity)

    return false unless entity.is_a?(Sketchup::Group)
    return false unless entity.attribute_dictionary ATTR_DICT
    
    true
    
  end
  
  # Get next serial number unique for a serialization name to use on a group/
  # component instance in structure.
  #
  # Method modified model by writing attribute, call it inside an operator.
  #
  # serial_name - Name of the part, e.g. "Pillar Double".
  #
  # Returns int.
  def self.next_serial_number(serial_name)
  
    model = Sketchup.active_model
    key = "structure_serial_count_#{serial_name}"
    serial_number = model.get_attribute ID, key
    serial_number = serial_number ? serial_number + 1 : 1
    model.set_attribute ID, key, serial_number
    
    serial_number
    
  end
  
  def self.properties_dialog(strs)
    #Open properties dialog for structures.
    #strs can either be one structure or an array of structures.
    
    strs = [strs] if Structure === strs
    
    #Create web dialog
    title = "#{S.tr("Structure Properties")} (#{strs.length} #{strs.length == 1 ? S.tr("Structure") : S.tr("Structures")})"
    dlg = UI::WebDialog.new title, false, "#{ID}_structures_properties", 330, 400, 300, 100, false
    dlg.navigation_buttons_enabled = false
    dlg.set_background_color dlg.get_default_dialog_color
    dlg.set_file(File.join(DIALOG_DIR, "structure_properties", "index.html"))
    
    structure_types = Template.list_installed :structure_type
    
    # Get values
    # If values differs between these structures, use nil as Indeterminate value.
    # If any of the used types is missing, warn user and try using another.
    values = {}
    #List all variable names here:
    ["structure_type", "disable_drawing"].each do |v_name|
      a = strs.map{ |t| t.send v_name }.uniq
      if v_name == "structure_type" && a != a & structure_types.map { |st| st[:id] }
        UI.messagebox S.tr("Used structure type could not be found. Applying changes will result in redrawing the structure as another structure type.")
        a = a & structure_types.map { |st| st[:id] }
        a = a.empty? ? [track_types[0][:id]] : [a.first]
      end
      values[v_name] = a.size == 1 ? a.first : nil
    end
    
    #Create list of structure types
    track_types = Template.list_installed :track_type
    structure_types.unshift({:name => "", :id => "", :info => ""}) if values["structure_type"] == nil
    structure_types.each do |st|
      #Make human readable strings from inch measurements
      if !st[:track_parallel_distance] || st[:track_parallel_distance] == 0 || st[:track_parallel_distance] == ""
        st.delete :track_parallel_distance
      else
        st[:track_parallel_distance] = st[:track_parallel_distance].to_l.to_s
      end
      #Show name, not ID of corresponding track
      if !st[:corresponding_track_type] || st[:corresponding_track_type] == ""
        st.delete :corresponding_track_type
      else
        st[:corresponding_track_type] = track_types.find{ |tt| tt[:id] == st[:corresponding_track_type] }[:name]
      end
    end
    structure_types_json = EneRailroad.object_2_json structure_types
    
    #Translate strings
    js = S.tr_dlg
    
    #Form data
    js << "var structure_types=#{structure_types_json};"
    js << "window.init_dynamic_lists();"
    values.each_pair do |k, v|
      js << "document.getElementById('#{k}').value='#{v.to_s}';"#nil > "", true > "true", false > "false"
    end
    
    #Init scripts
    js << "init_checkboxes();"
    js << "window.update_structure_type_info();"
    js << "document.getElementById('structure_type').onchange = update_structure_type_info;"
    
    #Show dialog
    if WIN
      dlg.show { dlg.execute_script js }
    else
      dlg.show_modal { dlg.execute_script js }
    end
    
    #Apply properties
    dlg.add_action_callback("apply") { |_, callbacks|

      #Get values from dialog
      #"reverse" is added as key now since it only exists locally.
      new_values = {}
      (values.keys << "reverse").each do |k|
        i = dlg.get_element_value(k)
        i = nil if i == ""
        new_values[k] = i
      end
      
      #Make boolean
      bool_vars = ["disable_drawing", "reverse"]
      bool_vars.each do |bv|
        next if new_values[bv] == nil
        new_values[bv] = new_values[bv] == "true"
      end
      
      #Save values and redraw
      
      #Disable observers
      #Start operation
      Observers.disable
      mod = Sketchup.active_model
      mod.start_operation "Structure Properties", true
      
      strs.each do |str|
      
        #Add new values to track
        new_values.each_pair do |k, v|
          next if v == nil#Nil means indeterminate, keep existing.
          next if k == "reverse"#reverse is not saved as a attribute
          str.instance_variable_set("@" + k, v)
        end
        
        str.reverse! if new_values["reverse"]
        
        str.draw
      
      end
    
      mod.commit_operation
      Observers.enable

      #Close if user pressed OK rather than Apply
      dlg.close if callbacks == "close"
      
    }

    #Close web dialog
    dlg.add_action_callback("close") {
      dlg.close
    }
    
  end
  
  # List parts hashes indexed by serial_number as hash indexed serial_name.
  #
  # Returns 2d hash.
  def self.list_parts
  
    # REVIEW: copying a part or whole structure results in duplicated serial_number. Perhaps assign new serial_number to those here?
  
    # List all parts.
    parts = {}
    Sketchup.active_model.entities.each do |g|
      next unless group_is_structure? g
      g.entities.each do |e|
        next unless [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class)
        next unless serial_name = e.get_attribute(Template::ATTR_DICT_PART, "serial_name")
        parts[serial_name] ||= {}
        serial_number = e.get_attribute(Template::ATTR_DICT_PART, "serial_number")
        parts[serial_name][serial_number] = e
      end
    end

    parts
    
  end
  
  # Transform one or more arrayed parts inside structures.
  #
  # serial_numbers must be longer or equally long as other arrays.
  # If serial_numbers is longer than other arrays the extra numbers will be
  # paired up with the first serial_name and transformation.
  #
  # serial_name     - Name of the part to move or an array of names.
  # serial_number   - Number of the the part to move or an array of numbers.
  # transformations - Transformation  to apply or array of transformations.
  #
  # Returns nothing.
  def self.transform_part(serial_names, serial_numbers, transformations)
  
    serial_names    = [serial_names]    unless serial_names.is_a?(Array)
    serial_numbers  = [serial_numbers]  unless serial_numbers.is_a?(Array)
    transformations = [transformations] unless transformations.is_a?(Array)
    
    raise ArgumentError("serial_numbers must be the longest array.") if serial_names.size >    serial_numbers.size
    raise ArgumentError("serial_numbers must be the longest array.") if transformations.size > serial_numbers.size
    
    part_list = list_parts
    
    serial_numbers.each_with_index do |serial_number, i|
      transformation = transformations[i] || transformations.first
      serial_name    = serial_names[i]    || serial_names.first
      e = part_list[serial_name][serial_number]
      e.transform! transformation# REVIEW: parents transformation may be different from unity transformation
      
      # Disable automatic redrawing (yes, this code is f*****g spaghetti! GAH!)
      ad = e.parent.instances.first.attribute_dictionary(Structure::ATTR_DICT)
      ad["disable_drawing"] = true unless ad["disable_drawing"]
    end
    
    nil
    
  end
  
  # Show dialog listing all serialized parts.
  #
  # Returns nothing.
  def self.part_info_dialog
  
    title = S.tr "Structure Part Info"
    dlg = UI::WebDialog.new title, false, "#{ID}_structures_part_info", 600, 400, 100, 100, true
    dlg.navigation_buttons_enabled = false
    dlg.set_background_color dlg.get_default_dialog_color
    dlg.set_file(File.join(DIALOG_DIR, "structure_part_info", "index.html"))
    
    part_list_html = lambda do
      parts = list_parts
      if parts
        html = "<table>"
        html << "<tr><td>Name</td><td>Number</td><td>X</td><td>Y</td><td>Z</td><td>Angle</td><td>Lat Long</td><td></td></tr>"
        parts.sort.each do |type|
          name, hash = type
          hash.sort.each do |part|
            number, e = part
            x = e.transformation.origin.x
            y = e.transformation.origin.y
            z = e.transformation.origin.z# REVIEW: parent might not have unity matrix transformation.
            v = Geom::Vector3d.new 1, 0, 0
            a = MyGeom.angle_in_plane v.transform(e.transformation), v
            html << "<tr><td>#{name}</td><td>#{number}</td><td>#{x}</td><td>#{y}</td><td>#{z}</td><td>#{Sketchup.format_angle a}</td><td>-------------</td><td><a href=\"#\" onclick=\"window.location='skp:move@#{name}|||||||#{number}';\">Move</a></td></tr>"# TODO: test with double and single quotes in name.
          end
        end
        html << "</table>"
      else
        html = S.tr "No serialized parts found in model's structures."# TODO: link to docs and tell what a serialized part is.
      end
      html
    end
    
    js = S.tr_dlg
    js << "document.getElementById('content').innerHTML=#{part_list_html.call.inspect};"
    if WIN
      dlg.show { dlg.execute_script js }
    else
      dlg.show_modal { dlg.execute_script js }
    end
    
    #Change style of web dialog into a toolbox (windows only)
    WinApi.dialog2toolbox
    
    dlg.add_action_callback("close") {
      dlg.close
    }
    
    dlg.add_action_callback("move") { |_, callback|
      name, number = callback.split "|||||||"# HACK: someone could put this string inside the name
      # Ignore angle since rotation point is old origin and getting that here gives more spaghetty code.
      input = UI.inputbox ["Delta X", "Delta Y", "Delta Z"], ["0", "0", "0"], S.tr("Move")
      next unless input
      number = number.to_i
      begin
        x = input[0].to_l
        y = input[1].to_l
        z = input[2].to_l
      rescue ArgumentError
        UI.messagebox S.tr "Could not parse as length."
      end
      t = Geom::Transformation.new Geom::Point3d.new(x, y, z)
      next if MyGeom.identity_matrix? t
      transform_part name, number, t
      js = "document.getElementById('content').innerHTML=#{part_list_html.call.inspect};"
      dlg.execute_script js
    }
    
    dlg.add_action_callback("export") {
      export_part_info
    }
    
    dlg.add_action_callback("import") {
      import_part_info
      js = "document.getElementById('content').innerHTML=#{part_list_html.call.inspect};"
      dlg.execute_script js
    }
    
    nil
    
  end
  
  def self.part_info_to_csv
  
    parts = list_parts
    csv = ""
    parts.sort.each do |type|
      name, hash = type
      hash.sort.each do |part|
        number, e = part
        name = name.gsub '"', '""'
        x = e.transformation.origin.x
        y = e.transformation.origin.y
        z = e.transformation.origin.z# REVIEW: parent might not have unity matrix transformation.
        v = Geom::Vector3d.new 1, 0, 0
        a = MyGeom.angle_in_plane v.transform(e.transformation), v
        csv << "\"#{name}\",#{number},#{x.to_m},#{y.to_m},#{z.to_m},#{a}\n"
      end
    end
    
    csv
      
  end
  
  def self.export_part_info
  
    path = UI.savepanel S.tr("Export"), @@export_path, "structure_parts.csv"
    return unless path
    File.open(path, "w") { |f| f.write part_info_to_csv }
    @@export_path = File.dirname path

    nil
    
  end
  
  def self.import_part_info
  
    path = UI.openpanel S.tr("Export"), @@export_path
    return unless path
    unless path.split(".").last.downcase == "csv"
      UI.messagebox "Wrong file format. Only .csv are allowed."
      return
    end
    
    # List existing part instances in hash indexed by
    # [serial_name, serial_number].
    existing = {}
    list_parts.each_pair do |name, hash|
      hash.each_pair do |number, e|
        existing[[name, number]] = e
      end
    end
    
    # List imported transformations in the same matter.
    imported = {}
    regex = /^"(.*)",(\d+),(-?\d+\.?\d*),(-?\d+\.?\d*),(-?\d+\.?\d*),(-?\d+\.?\d*)/
    IO.foreach(path) do |l|
      mathces = regex.match l
      _, name, number, x, y, z, a = mathces.to_a
      number = number.to_i
      x = x.to_f.m
      y = y.to_f.m
      z = z.to_f.m
      a = a.to_f
      t = Geom::Transformation.new Geom::Point3d.new(x, y, z)
      t *= Geom::Transformation.rotation ORIGIN, Z_AXIS, a
      imported[[name, number]] = t
    end

    # Compare lists.
    if (existing.keys - imported.keys).size > 0
      UI.messagebox S.tr "Model contains serialized parts missing in imported file. These will be ignored."
    end
    if (imported.keys - existing.keys).size > 0
      UI.messagebox S.tr "Imported file contains parts missing in model. These will be ignored"
    end
    keys = imported.keys && existing.keys
    
    model = Sketchup.active_model
    model.start_operation S.tr "Import Structure Part Data"

    serial_names    = []
    serial_numbers  = []
    transformations = []
    keys.each do |k|
      name    = k[0]
      number  = k[1]
      t_old   = existing[k].transformation# REVIEW: based on the assumption that structure group uses unity matrix as transformation.
      t_new   = imported[k]
      t_delta = t_new * (t_old.inverse)
      
      # skip if part hasn't moved.
      next if MyGeom.identity_matrix? t_delta
      
      serial_names    << name
      serial_numbers  << number
      transformations << t_delta
    end
    transform_part serial_names, serial_numbers, transformations
    
    model.commit_operation
    
  end
  
  #Instance attribute accessors

  attr_accessor :path
  attr_accessor :structure_type
  attr_accessor :group
  attr_accessor :disable_drawing

  #Instance methods

  def initialize(group = nil)
    #Create new structure object, either default(empty) or load from group.
    #Declare all structure's variables.

    if group
      #Create object from existing data saved in group
      #Used to re-initialize saved structure with data saved to attributes, NOT create new structure from any group.

      group.attribute_dictionary(ATTR_DICT).each_pair do |key, value|
        self.instance_variable_set("@" + key.to_s, value)
      end

      #Override group reference since group cannot be saved as attribute
      @group = group

    else
      #Create new object with blank settings.

      @path = []

      @structure_type = nil

      #Group to draw structure and save data to.
      #Structure groups are always on model root.
      #Created and drawn by Structure.draw.
      #Observers attached there too.
      @group = nil

      #Warn before redrawing. Prevents manual changes from being lost
      @disable_drawing = false

    end

    #Add structure to structure list
    #@@instances << self

  end

  def draw
    #Draw the structure to its group according to path and structure_type
  
    #Create group if there isn't one
    unless @group
      @group = Sketchup.active_model.entities.add_group
      @group.add_observer(Observers::MyInstanceObserver.new)
      #@group.add_observer(Observers::MyEntityObserver.new)
    end
    
    # REVIEW: this code is based on the assumption that group has unity transformation.
    
    #Make structure unique
    @group.name += ""#Group.make_unique is deprecated and this seems to work instead
    
    #Standard vars
    defs = model.definitions
    ents_group = @group.entities
    
    ents_group.clear!
    
    #Remove duplicated points right after each other in path.
    i = 0; @path.delete_if { |p| i+=1; i>0 && p == @path[i] }
    
    #Get end transformations
    trans_ends = MyGeom.calc_trans_ends @path
    
    component_def = Template.component_def :structure_type, @structure_type
    unless component_def
      warn "Could not draw structure. Structure type '#{@structure_type}' could not be found.\n#{caller}"
      return
    end
        
    # Make extrusion of there's a profile for it.
    profile_template = component_def.entities.find do |e|
      e.get_attribute(Template::ATTR_DICT_PART, "type") == "extrusion_profile"
    end
    if profile_template

      # Place profile at track end.
      profile_component_def = #TODO: get definition from method on Template.
        if profile_template.is_a?(Sketchup::Group)
          profile_template.entities.parent
        else
          profile_template.definition
        end
      component_inst = ents_group.add_instance profile_component_def, trans_ends[0]

      #Explode profile component. should only contain groups containing one face each.
      ents_exploded = component_inst.explode
      ents_exploded.each do |g|

        #Keep original group in component definition so component doesn't change and have to be reloaded each time a structure of this type is drawn
        g.name += ""#Group.make_unique is deprecated and this seems to work instead

        #Copy and adapt path to be relative to the group for this individual extrusion face
        trans_correction = g.transformation.inverse
        correcetd_path = @path.map { |p| p.clone.transform trans_correction }

        #Extrude face
        face = g.entities.find { |i| i.class==Sketchup::Face }
        endFace = EneRailroad.extrude(face, correcetd_path, Z_AXIS)

        #Hide ends of extrusion
        face.hidden = endFace.hidden = true
        (face.edges + endFace.edges).each { |i| i.hidden = true }

      end#each

    end#if file exists
      
    #Add arrayed elements
    array_templates = component_def.entities.select do |e|#TODO: get definition from method on Template.
      e.get_attribute(Template::ATTR_DICT_PART, "type") == "arrayed"
    end
    array_templates.each do |array_template|

      distance = array_template.get_attribute(Template::ATTR_DICT_PART, "distance", 1.m)

      arrayed_component_def =
        if array_template.is_a?(Sketchup::Group)
          array_template.entities.parent
        else
          array_template.definition
        end
        
      override_half_end_distance = array_template.get_attribute(Template::ATTR_DICT_PART, "override_half_end_distance", false)

      trans_along = MyGeom.calc_trans_along_path @path, distance, override_half_end_distance
      trans_along.each do |i|
        instance = ents_group.add_instance arrayed_component_def, i 
        if serial_name = array_template.get_attribute(Template::ATTR_DICT_PART, "serial_name")
          instance.set_attribute(Template::ATTR_DICT_PART, "serial_name", serial_name)
          serial_number = self.class.next_serial_number serial_name
          instance.set_attribute(Template::ATTR_DICT_PART, "serial_number", serial_number)
        end
      end
    end
    
    #Add endings
    end_template = component_def.entities.find do |e|
      e.get_attribute(Template::ATTR_DICT_PART, "type") == "ending"
    end
    if end_template
      end_component_def = #TODO: get definition from method on Template.
        if end_template.is_a?(Sketchup::Group)
          end_template.entities.parent
        else
          end_template.definition
        end
      0.upto(1) do |i|
        instance = ents_group.add_instance end_component_def, trans_ends[i+2]
        #if serial_name = end_template.get_attribute(Template::ATTR_DICT_PART, "serial_name")
        #  instance.set_attribute(Template::ATTR_DICT_PART, "serial_name", serial_name)
        #  serial_number = self.class.next_serial_number serial_name
        #  instance.set_attribute(Template::ATTR_DICT_PART, "serial_number", serial_number)
        #  if end_template.get_attribute(Template::ATTR_DICT_PART, "count_overlapping_as_one")
        #    instance.set_attribute(Template::ATTR_DICT_PART, "count_overlapping_as_one", true)
        #  end
        #end
      end
    end
    
    #Save data as attributes
    self.save_attributes
    
  end

  def inspect
    #Do not show attributes when inspecting. It clutters the console
    self.to_s
  end

  def length
    #Returns length of structure.
    
    MyGeom.path_length @path
    
  end
  
  # Public: Model structure is drawn to or active Model if structure has not
  # been drawn.
  #
  # Returns a Model.
  def model
  
    @group && @group.valid? ? @group.model : Sketchup.active_model
    
  end
  
  def reverse!
    #Reverse path.
    #Useful when asymmetrical structure, e.g. a double track bridge or a platform has been placed on the wrong side of the track.

    @path.reverse!
    
    true

  end

  def save_attributes
    #Writes all instance variables as attributes to the group so it can be retrieved when re-initializing object.
    #Called from draw.

    #Do not alter model when called from observer triggered undo/redo.
    return if Observers.undo_redo?
    
    #Add all class instance variables as group attributes
    self.instance_variables.each do |i|
      key = i
      value = self.instance_variable_get(key)
      key = key.to_s  #Make string of symbol
      key[0] = ""    #Remove @ sign from key
      next if key == "group"
      @group.set_attribute ATTR_DICT, key, value
    end#each

    @group.name = "#{S.tr("Structure")} #{length}"

  end
  
end#class

end#module