# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolStructurePosition
  #This tool is used to move structures.
    
  #Length to lock path to
  #This togglable behavior makes both ends slide when one is moved to keep the length of the path.
  #Toggle with alt.
  @@length_locked_to = nil
  
  #Sketchup tool definitions
  
  def initialize(str = nil)
  
    #Path being edited
    @path = []
    
    #Start and end of path
    @controls = []
    
    #Track to lock to
    #When hovering a track the user can hold down shift to lock start/end position to that track and
    #actually clicking closer to another track, e.g. aligning start/end with the start/end of a structure
    #on a parallel track.
    @track_locked_on = nil
  
    #Try to get structure being positioned from selection if not defined.
    ss = Sketchup.active_model.selection
    str = Structure.get_from_group ss[0] if !str && ss.length == 1
    
    #Structure being positioned
    @str = str
    
    #Hide structure so path can be seen.
    load_structure str if str
    
    #Save screen coords for latest click and compare mouse location to when releasing to see if points were dragged
    @x_down = nil
    @y_down = nil
    
    #Index of point that is moved by the cursor.
    #0 for path start, 1 for path end.
    @control_moving = nil

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    #Statusbar texts
    @status_text_select = S.tr "Select a structure to position."
    @status_text_edit = S.tr "Move structure end points. Enter = Save, Alt = Toggle keep length."
    @status_text_move = S.tr "Click to place end or type in exact length. Shift = Lock on hovered track, Alt = Toggle keep length."
    @vcb_label = S.tr "Length"
    @vcb_value = nil
    
    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "structure_position.png"), 2, 2)
    @cursor_fixed_length = UI.create_cursor(File.join(CURSOR_DIR, "structure_position_fixed_length.png"), 2, 2)
    
  end
  
  def onSetCursor

    UI.set_cursor @@length_locked_to? @cursor_fixed_length : @cursor#NOTE: SU ISSUE: doesn't update on key down. waits until user moves cursor.

  end
  
  def enableVCB?
  
    @control_moving
    
  end
  
  def activate

    #Reset statusbar text
    @status_text = @str ? @status_text_edit : @status_text_select
    Sketchup.set_status_text @status_text, SB_PROMPT
    Sketchup.set_status_text @vcb_label, SB_VCB_LABEL
    Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE

  end

  def deactivate(view)

    save_structure @str if @str

  end
  
  def draw(view)
  
    color = Sketchup::Color.new(255, 194, 42)
  
    if @str
      
      #Draw path
      view.line_width = 1
      view.drawing_color = Sketchup::Color.new("White")
      MyView.draw_polyline_on_top(view, @path) if @path
      
      #Draw points
      view.draw_points @controls, 10, 2, color
      
      #Draw input point
      @ip.draw view if @ip.display? && @control_moving
      
    end
    
  end
  
  def onReturn(view)
    #Redraw selected structure to model and go back to structure selecting mode

    save_structure @str if @str

    @str = nil
    @control_moving = nil
    @track_locked_on = nil
    view.invalidate

    #Reset statusbar text
    @status_text = @status_text_select
    Sketchup.set_status_text @status_text, SB_PROMPT
    @vcb_value = ""
    Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE

  end
 
  def onCancel(flag, view)
    #Go back to structure selecting mode without altering structure

    if @str
      Observers.disable
      @str.model.start_operation "Track Position", true

      @str.group.entities.clear!#Clear edge that prevented group from being removed
      @str.draw

      @str.model.commit_operation
      Observers.enable
    end

    @str = nil
    @control_moving = nil
    @track_locked_on = nil
    view.invalidate

    #Reset statusbar text
    @status_text = @status_text_select
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    @vcb_value = ""
    Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE
    
  end
  
  def onMouseMove(flags, x, y, view)
  
    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y
    
    #When in select mode, temporarily make the hovered structure selected in model to outline it
    unless @str
      view.model.selection.clear
      if picked && Structure.group_is_structure?(picked)
        view.model.selection.add picked
      end
    end
    
    #If a control is being moved, re-calculate path
    if @control_moving
      point = @ip.position
      q = Track.inspect_point @ip.position, nil, @track_locked_on
      point = q[:point]
      @controls[@control_moving] = point
      @path = Track.path_between_points(@controls[0], @controls[1])
      @path = @path[:path] if @path
      
      if @path
      
        #If length is set to be constant, move other end of path too.
        if @@length_locked_to
          current_length = MyGeom.path_length @path
          missing_length = @@length_locked_to - current_length
          
          #If path end is moved, temporarily reverse it so it's truncated/added to in the right end.
          @path.reverse! if @control_moving == 1
      
          if missing_length < 0
            #To long, make shorter
            @path = MyGeom.path_truncate @path, @@length_locked_to
          else
            #To short, make longer
            @path += Track.extract_path(@path[-1], missing_length, @path[-1] - @path[-2])[1..-1]
          end
          
          @path.reverse! if @control_moving == 1
          
          #Update controls to fit path
          @controls = []
          @controls << @path[0]
          @controls << @path[-1]
          
        else
          #Not constant length, update vcb
          
          @vcb_value = MyGeom.path_length @path
          Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE
          
        end
        
      else
        #No path found
        @vcb_value = ""
        Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE
      end
      
      view.invalidate
    end
    
  end
  
  def onLButtonDown(flags, x, y, view)
  
    @x_down = x
    @y_down = y

    @ph.do_pick(x, y)
    picked = @ph.best_picked

    if !@str
      #In select mode, selected clicked structure
      #yeah, I know if ! else is odd but it's the order these actions are performed in and therefore makes sense in the code
      
      if picked
        str = Structure.get_from_group(picked)
        if str
          load_structure str
          view.invalidate
          @status_text = @status_text_edit
          Sketchup.set_status_text(@status_text, SB_PROMPT)
        end
      end
      
    else
      #In edit mode
      
      if !@control_moving
        #start moving path if path end is clicked
        
        0.upto(1) do |i|
          if EneRailroad.mouse_on_point?(view, x, y, @controls[i])
            @control_moving = i
            break
          end
        end
        if @control_moving
          @status_text =  @status_text_move
          Sketchup.set_status_text(@status_text, SB_PROMPT)
        end
        
      else
        #Stop moving path
        
        @ip.clear
        @control_moving = nil
        @track_locked_on = nil
        view.invalidate

        @status_text = @status_text_edit
        Sketchup.set_status_text @status_text, SB_PROMPT
      
      end
    
    end
    
  end
  
  def onLButtonUp(flags, x, y, view)

    if (@x_down - x).abs > 10 && (@y_down - y).abs > 10
      #Stop moving point, same code as on mouse down
      #Mouse has to be moved 10px in x or y for it to count as dragged

      @ip.clear
      @control_moving = nil
      @track_locked_on = nil
      view.invalidate

      @status_text = @status_text_edit
      Sketchup.set_status_text @status_text, SB_PROMPT
    end

  end
  
  def onKeyDown(key, repeat, flags, view)

    #Lock on track (closest)
    if key == CONSTRAIN_MODIFIER_KEY
      #picked = @ph.best_picked
      #t = Track.get_from_group picked
      q = Track.inspect_point @ip.position, nil
      t = q[:track]
      @track_locked_on = t if t
      return true
    end
    
    #Toggle lock length
    if key == ALT_MODIFIER_KEY
      if !@@length_locked_to && @str
        @@length_locked_to = MyGeom.path_length @path#NOTE: use length of existing structure instead of path currently being modified? @str.path instead of @path. then VCB needs to be updated too
      else
        @@length_locked_to = nil
      end
      return true
    end

  end

  def onKeyUp(key, repeat, flags, view)
  
    #Unlock from track
    if key == CONSTRAIN_MODIFIER_KEY
      @track_locked_on = nil
      return true
    end
    
  end
  
  def onUserText(text, view)
  
    return unless @control_moving
    return unless @path
    
    begin
      distance = text.to_l
    rescue
      UI.messagebox(S.tr("Invalid length."))
      return
    end
    
    #If tool is locked to length, update that length
    @@length_locked_to = distance if @@length_locked_to    
    
    current_length = MyGeom.path_length @path
    
    #If path start is moved, temporarily reverse it so it's truncated/added to in the right end.
    @path.reverse! if @control_moving == 0
    
    if distance < current_length
      #Draw shorter than current path, truncate
      @path = MyGeom.path_truncate @path, distance
      
    else
      #Draw longer, start at path end and calculate new
      missing_length = distance - current_length
      @path += Track.extract_path(@path[-1], missing_length, @path[-1] - @path[-2])[1..-1]
    end
    
    @path.reverse! if @control_moving == 0
    
    #Update controls to fit path
    @controls = []
    @controls << @path[0]
    @controls << @path[-1]
    
    #Stop moving, same code as in onLButtonDown
    @ip.clear
    @control_moving = nil
    @track_locked_on = nil
    view.invalidate

    @status_text = @status_text_edit
    Sketchup.set_status_text @status_text, SB_PROMPT

  end
  
  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    
    Sketchup.set_status_text @status_text, SB_PROMPT
    Sketchup.set_status_text @vcb_label, SB_VCB_LABEL
    Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE
    view.invalidate
    
  end
  
  #Own definitions
  #Not called from Sketchup itself
  
  def load_structure(str)
    #Load structure data and empty structure group so path can be seen
    
    if str.disable_drawing
      msg = S.tr("This structure is set to not be automatically redrawn. It may contain manually made changes that will be lost if redrawn.") + "\n\n" + S.tr("Change position and redraw anyway?")
      if UI.messagebox(msg, MB_YESNO) == IDYES
        str.disable_drawing = false
        str.model.start_operation(S.tr("Enable Automatic Structure Drawing"), true)
        str.save_attributes
        str.model.commit_operation
      else
        return
      end
    end
    
    structure_types = Template.list_installed :structure_type
    unless structure_types.map { |st| st[:id] }.include? str.structure_type
      msg = S.tr("Used structure type could not be found. Applying changes will result in redrawing the structure as another structure type.")
      if UI.messagebox(msg, MB_OKCANCEL) == IDOK
        str.structure_type = structure_types[0][:id]
      else
        return
      end
    end
  
    @str = str
    @path = str.path.dup
    @controls = []
    @controls << @path[0]
    @controls << @path[-1]
    
    #If locking to length, update length to lock to
    @@length_locked_to = MyGeom.path_length @path if @@length_locked_to
    
    #Write current distance to VCB
    @vcb_value = @path ? MyGeom.path_length(@path) : ""
    Sketchup.set_status_text @vcb_value.to_s, SB_VCB_VALUE
    
    #Transparent operation so when it's undone at the same time as the next - redrawing when saving track
    str.model.start_operation(S.tr("Structure Position"), true, true, false)

    #Empty group so control points can be seen
    str.group.entities.clear!
    
    #Add a random edge so Sketchup doesn't delete the group
    str.group.entities.add_edges Geom::Point3d.new, Geom::Point3d.new(1, 0, 0)

    str.model.commit_operation
    
  end
  
  def save_structure(str)
    #Save path and redraw
    
    return unless @path
    str.path = @path.dup

    # HACK: Preload component definitions before starting operation not to break
    # it in older SU versions.
    if Sketchup.version < "14"
      Template.component_def :structure_type, str.structure_type
    end
    
    str.model.start_operation "Structure Position", true
    Sketchup.set_status_text S.tr("Drawing Structure"), SB_PROMPT

    str.group.entities.clear!#Clear edge that prevented group from being removed
    str.draw

    Sketchup.set_status_text(S.tr("Done Drawing Structures"), SB_PROMPT)
    str.model.commit_operation
  
  end
  
end#class

end#module