# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolTrackPosition
  #This tool is used to change the control points and vectors of existing Tracks
  
  #Sketchup tool definitions

  def initialize(t = nil)

    #Control points & vectors (uses own variable instead of @t.controls since controls may not be sved)
    @controls = []
    #  0  Point where track starts
    #  1  Point where track ends
    #  2  vector from start point
    #  3  vector from end point

    @curve_algorithm = nil

    #Index of point that is moved by the cursor.
    #If start or end is being moved its corresponding control point is moved too
    @control_moving = nil

    #Key toggles
    @alt_toggle = false

    #Initialize pick helper to select track
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to move control points with
    @ip = Sketchup::InputPoint.new

    #Reposition selected track if a track is selected and track isn't set as an argument on initializing
    ss = Sketchup.active_model.selection
    t = Track.get_from_group ss[0] if !t && ss.length == 1

    load_track(t) if t

    #Save screen coords for latest click and compare mouse location to when releasing to see if points were dragged
    @x_down = nil
    @y_down = nil

    #Statusbar texts
    @status_text_select = S.tr "Select a track to position."
    @status_text_edit = S.tr "Move control points to position track. Enter = Save, Right Click = Options."
    @status_text_move_point = S.tr "Place control point on a track to connect this to it. Shift = Lock to straight."
    @status_text_move_vector = S.tr "Alt = Toggle direction lock."
    @vcb_label = "Length[;Radius]"

    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "track_position.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def enableVCB?
  
    self.moving_arc
 
  end
  
  def activate

    #Reset statusbar text
    @status_text = @t ? @status_text_edit : @status_text_select
    Sketchup.set_status_text(@status_text, SB_PROMPT)

  end

  def deactivate(view)

    save_track(@t) if @t

  end

  def draw(view)

    if @t
      #Draw control points and path to screen

      color = Sketchup::Color.new(162, 162, 255)

      #Draw info text
      text = S.tr("Curve Algorithm") + ": "
      text <<  case @curve_algorithm
               when "hard_corner"
                 S.tr "Sharp Corner"
               when "c_bezier"
                 S.tr "BÃ©zier (Cubic)"
               when "arc"
                 S.tr "Arc"
               end
      text << " "
      text << S.tr("(Right click to change)")
      view.draw_text [4, 0, 0], text

      #Draw path
      view.line_width = 1
      view.drawing_color = Sketchup::Color.new("Black")
      if @controls[2].valid? && @controls[3].valid? && @controls[0] != @controls[1]
        path = MyGeom.calc_path @controls, @curve_algorithm
        unless (MyGeom.flatten_vector(path[-2]-(path[-1]))).samedirection?(MyGeom.flatten_vector(@controls[3]))#NOTE: precision problem. when very very close to straight but not entirely there's an error here in same_direction?.
          #End segment isn't parallel to end vector in x-y plane
          view.drawing_color = Sketchup::Color.new("Red")
          view.line_stipple = "-"
          @tooltip = S.tr "Can't draw an arc. Try BÃ©zier curve instead"#NOTE: SU ISSUE: make text red when possible
        end
        view.draw_polyline path
      end
      view.line_stipple = ""

      #Draw lines
      view.line_width = 3
      view.drawing_color = color
      0.upto(1) do |i|
        view.draw_line @controls[i], @controls[i].offset(@controls[i+2])
      end

      #Draw points
      0.upto(1) do |i|
        view.draw_points [@controls[i]], 10, 2, color
      end

      #Draw vector points
      [2, 3].each do |i|
        view.draw_points [@controls[i-2].offset(@controls[i])], 8, 2, color
        view.draw_points [@controls[i-2].offset(@controls[i])], 7, 2, Sketchup::Color.new("white")
      end

      #Draw input point
      @ip.draw view if @ip.display? && @control_moving

      #Draw tooltip
      view.tooltip = @tooltip ? @tooltip: @ip.tooltip

    end

  end

  def onReturn(view)
    #Redraw selected track to model and go back to track selecting mode

    save_track(@t) if @t

    @t = nil
    @control_moving = nil
    view.invalidate

    #Reset statusbar text
    @status_text = @status_text_select
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.set_status_text(nil, SB_VCB_LABEL)

  end

  def onCancel(flag, view)
    #Go back to track selecting mode without altering track

    if @t
      @t.model.start_operation "Track Position", true

      @t.group.entities.clear!#Clear edge that prevented group from being removed
      @t.queue_drawing
      @t.queue_changed_connections
      Track.draw_queued

      @t.model.commit_operation
    end

    @t = nil
    @control_moving = nil
    view.invalidate

    #Reset statusbar text
    @status_text = @status_text_select
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    Sketchup.set_status_text(nil, SB_VCB_LABEL)

  end

  def onMouseMove(flags, x, y, view)

    @ph.do_pick(x, y)
    picked = @ph.best_picked
    @ip.pick view, x, y

    @tooltip = nil

    unless @t
      #If no track is selected, temporarily select the hovered in model

      view.model.selection.clear
      if picked && Track.group_is_track?(picked)
        view.model.selection.add picked
      end#if
    end#unless

    if @control_moving
      #Moving control points
      #If start or end moves, move corresponding control point too
      #If start or end moves onto an existing track, use point and vector from closest end of that track

      if @control_moving < 2
        #Start or end point is being moved
        
        #More  readable names for control points and vectors indexes
        #@control_moving is the point being moved
        corresponding_vector = @control_moving + 2
        other_point =  1 - @control_moving
        other_vector = other_point + 2
        
        point = @ip.position
        point_cam = view.camera.eye
        line_to_cam = [point, point_cam - point]
        line_straight_track = [@controls[other_point], @controls[other_vector]]
      
        #When shift is pressed track is straight
        shift = (flags & CONSTRAIN_MODIFIER_MASK) == CONSTRAIN_MODIFIER_MASK
        if shift

          @controls[@control_moving] = point.project_to_line line_straight_track
          length = @controls[corresponding_vector].length
          @controls[corresponding_vector] = @controls[other_vector].reverse
          @controls[corresponding_vector].length = length
          view.invalidate
          return

        end#if

        if Track.group_is_track? picked
          #A track is hovered, attach track being positioned to it
          t_h = Track.get_from_group picked
          if t_h

            @tooltip = S.tr "Connect To Track"

            if t_h.controls[0].distance(point) < t_h.controls[1].distance(point)
              #Cursor closest to start

              @controls[@control_moving] = t_h.controls[0]
              length = @controls[corresponding_vector].length
              @controls[corresponding_vector] = t_h.controls[2].reverse
              @controls[corresponding_vector].length = length
              view.invalidate
              return

            else
              #Cursor closest to end

              @controls[@control_moving] = t_h.controls[1]
              length = @controls[corresponding_vector].length
              @controls[corresponding_vector] = t_h.controls[3].reverse
              @controls[corresponding_vector].length = length
              view.invalidate
              return

            end
          end
        end
        
        #If input point is in free space (not on an entity) it's placed on the x y, y z or z x planes
        #Instead use point on the horizontal plane going through the other track end point   
        if @ip.degrees_of_freedom == 3
          plane = [@controls[other_point], Z_AXIS]
          point = Geom.intersect_line_plane line_to_cam, plane
        end#if
        
        #Snap to straight (unless an endpoint is hovered)
        unless @ip.degrees_of_freedom == 0
          point_on_straight_track_closest_to_input = Geom.closest_points(line_straight_track, line_to_cam)[0]
          if EneRailroad.mouse_on_point?(view, x, y, point_on_straight_track_closest_to_input)
            @tooltip = S.tr "Straight track"
            point = point_on_straight_track_closest_to_input
            @ip.clear#Don't draw input point if behind the point track end is snapped to
          end#if
        end#unless
        
        @controls[@control_moving] = point

        #Set moving point's corresponding vector so an arc can be formed
        #Find vector to mirror on in x-y plane
        vector_betwwen_points = @controls[@control_moving]- @controls[other_point]
        vector_mirror = vector_betwwen_points.cross(Geom::Vector3d.new(0, 0, 1))
        #Mirror other vector and use as vector
        @controls[corresponding_vector] = @controls[other_vector].transform Geom::Transformation.rotation(Geom::Point3d.new, vector_mirror, Math::PI)

        view.invalidate

      else
        #Changing vector
        #Toggle direction lock with alt

        if @alt_toggle
          #Vector can change freely

          vector = @ip.position - @controls[@control_moving-2]
          @controls[@control_moving] = vector if vector.valid?
          view.invalidate
        else
          #Only changing length of vector

          locked_line = [@controls[@control_moving-2], @controls[@control_moving]]
          end_of_vector = @ip.position.project_to_line locked_line
          vector = end_of_vector - @controls[@control_moving-2]
          @controls[@control_moving] = vector if vector.valid?
          view.invalidate
        end
      end
    end

  end

  def onLButtonDown(flags, x, y, view)

    @ph.do_pick(x, y)
    picked = @ph.best_picked

    if !@t
      #Start editing clicked track

      if Track.group_is_track? picked
        t = Track.get_from_group picked
        if t
          load_track(t)

          if @t
            view.invalidate
            @status_text = @status_text_edit
            Sketchup.set_status_text(@status_text, SB_PROMPT)
          end
        end
      end
    else
      #Move control points

      if !@control_moving
        #Start moving clicked control point if any

        #Get clicked point
        (0..1).each do |i|
          if EneRailroad.mouse_on_point?(view, x, y, @controls[i])
            @control_moving = i
            break
          end
        end#each

        #Get clicked vector
        (2..3).each do |i|
          if EneRailroad.mouse_on_point?(view, x, y, @controls[i-2].offset(@controls[i]))
            @control_moving = i
            break
          end
        end#each

        if @control_moving
          @status_text = @control_moving < 2 ? @status_text_move_point : @status_text_move_vector
          Sketchup.set_status_text(@status_text, SB_PROMPT)
        end

        #Enable VCB label if end point of arc is moved
        Sketchup.set_status_text(@vcb_label, SB_VCB_LABEL) if self.moving_arc

      else
        #Stop moving point, same code as on mouse up

        @ip.clear
        @control_moving = nil
        view.invalidate

        @status_text = @status_text_edit
        Sketchup.set_status_text(@status_text, SB_PROMPT)
        Sketchup.set_status_text(nil, SB_VCB_LABEL)

      end

    end
    @x_down = x
    @y_down = y
  end

  def onLButtonUp(flags, x, y, view)

    if (@x_down - x).abs > 10 && (@y_down - y).abs > 10
      #Stop moving point, same code as on mouse down
      #Mouse has to be moved 10px in x or y for it to count as dragged

      @ip.clear
      @control_moving = nil
      view.invalidate

      @status_text = @status_text_edit
      Sketchup.set_status_text(@status_text, SB_PROMPT)
      Sketchup.set_status_text(nil, SB_VCB_LABEL)
    end

  end

  def onUserText(text, view)

    if self.moving_arc
      #Track control point is being moved and curve is arc.
      #Set track position to "<length>;<radius>".
      
      input = text.split(/;|, /)#Split at ";" or ", "
      input.each_with_index do |v, i|
        begin
          input[i] = v.to_l
        rescue
          UI.messagebox(S.tr("Invalid length."))
          return
        end
      end
      
      #p_moved = @controls[@control_moving]
      p_other = @controls[1-@control_moving]
      #v_moved = @controls[@control_moving+2]
      v_other = @controls[3-@control_moving]
      
      if input.length == 1 || input[1] == 0
        #Only length, make track straight
        @controls[@control_moving] = p_other.offset v_other, input[0]
        @controls[@control_moving+2] = v_other.reverse
      else
        #Length + radius
        c = p_other.offset(Z_AXIS*v_other, input[1])
        a = input[0] / input[1]
        t = Geom::Transformation.rotation c, Z_AXIS, a
        
        @controls[@control_moving] = p_other.transform t
        @controls[@control_moving+2] = v_other.transform(t).reverse
      end
      
      @control_moving = nil
      view.invalidate
    end

  end

  def getMenu(menu)

    if @t
      menu.add_item(S.tr("Set Control Points And Vectors...")) { self.modify_position_manually }
      menu.add_separator
      options = [
        ["hard_corner", S.tr("Sharp Corner")],
        ["c_bezier", S.tr("BÃ©zier (Cubic)")],
        ["arc", S.tr("Arc")]
      ]
      options.each do |i|
        item = menu.add_item(i[1]) {
          @curve_algorithm = i[0]
          Sketchup.active_model.active_view.invalidate
        }
        menu.set_validation_proc(item) {
          i[0] == @curve_algorithm ? MF_CHECKED : MF_ENABLED
        }
      end
    end

  end

  def onKeyDown(key, repeat, flags, view)

    #Change alt toggle variable that affects how vectors are repositioned
    if key == ALT_MODIFIER_KEY
      @alt_toggle = !@alt_toggle
      return true
    end

  end

  def resume(view)
    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate
  end

  #Own definitions
  #Not called from Sketchup itself

  def load_track(t)
    #Set track to edit and load its data
    
    if t.disable_drawing
      msg = S.tr("This track is set to not be automatically redrawn. It may contain manually made changes that will be lost if redrawn.") + "\n\n" + S.tr("Change position and redraw anyway?")
      if UI.messagebox(msg, MB_YESNO) == IDYES
        t.disable_drawing = false
        t.model.start_operation(S.tr("Enable Automatic Track Drawing"), true)
        t.save_attributes
        t.model.commit_operation
      else
        return
      end
    end
    
    track_types = Template.list_installed :track_type
    unless track_types.map { |tt| tt[:id] }.include? t.type_of_track
      msg = S.tr("Used track type could not be found. Applying changes will result in redrawing the track as another track type.")
      if UI.messagebox(msg, MB_OKCANCEL) == IDOK
        t.type_of_track = track_types[0][:id]
      else
        return
      end
    end
    
    if t.type_of_signals && !Template.list_installed(:signal_type).map { |st| st[:id] }.include?(t.type_of_signals)
      msg = S.tr("Used signal type could not be found. Applying changes will result in redrawing the track without signals.")
      if UI.messagebox(msg, MB_OKCANCEL) == IDOK
        t.type_of_signals = false
      else
        return
      end
    end
    
    @t = t

    @controls = t.controls.dup
    @curve_algorithm = t.curve_algorithm

    #Transparent operation so when it's undone at the same time as the next - redrawing when saving track
    t.model.start_operation(S.tr("Track Position"), true, true, false)

    #Empty group so control points can be seen
    t.group.entities.clear!
    #Add a random edge so Sketchup doesn't delete the group
    t.group.entities.add_edges Geom::Point3d.new, Geom::Point3d.new(1, 0, 0)

    t.model.commit_operation

  end

  def save_track(t)
    #Applies new data to track and redraws it

    t.controls = @controls.dup
    t.curve_algorithm = @curve_algorithm

    # HACK: Preload component definitions before starting operation not to break
    # it in older SU versions.
    if Sketchup.version < "14"
      Template.component_def :track_type, t.type_of_track
      Template.component_def :signal_type, t.type_of_signals if t.type_of_signals
    end
    
    t.model.start_operation "Track Position", true

    t.group.entities.clear!#Clear edge that prevented group from being removed
    t.queue_drawing
    t.queue_changed_connections
    Track.draw_queued

    t.model.commit_operation

  end

  def modify_position_manually
    #Prompt point & vector coordinates

    prompts = [S.tr("Start Point")+", X", "..Y", "..Z", S.tr("End Point")+", X", "..Y", "..Z", S.tr("Start Vector")+", X", "..Y", "..Z", S.tr("End Vector")+", X ", "..Y", "..Z"]
    defaults = [
      @controls[0][0], @controls[0][1], @controls[0][2],
      @controls[1][0], @controls[1][1], @controls[1][2],
      @controls[2][0].to_l, @controls[2][1].to_l, @controls[2][2].to_l,
      @controls[3][0].to_l, @controls[3][1].to_l, @controls[3][2].to_l
    ]
    title = S.tr "Track Position"
    input = UI.inputbox prompts, defaults, title
    return unless input
    
    #Only update coordinate if it's not so close to original value they would both translate to the same string.
    #This prevents values of unchanged fields from affecting position due to rounding.
    (0..input.length-1).each do |i|
      control = i/3
      coord = i%3
      @controls[control][coord] = input[i].to_l unless @controls[control][coord].to_l.to_s == input[i].to_l.to_s
    end
    
    Sketchup.active_model.active_view.invalidate

  end

  def moving_arc
    #True if a point (not vector) defining an arc curve is moved.
    
    @t && @control_moving && @control_moving < 2 && @curve_algorithm == "arc"
    
  end
  
end#class

end#module
