# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class Train
  #A train is a collection of connected rolling stocks
  #Train objects cannot be saved when the application closes but they are re-initialized from rolling stocks on model load.

  #All loaded train objects. In mac these might be in different models
  @@instances ||= []

  #Class variable accessors

  def self.instances; @@instances; end
  def self.instances=(value); @@instances = value; end

  #Class methods

  def self.get_from_name(name, model = nil)
    #Return train by name string
    #May be used in Balise to easier get the object of a known train.
    
    model ||= Sketchup.active_model
    
    @@instances.select { |i| i.name == name && i.model == model} [0]
    
  end
  
  def self.get_from_r_stock(rs)
    #Get the train a specific rolling stock belongs to

    @@instances.find { |t| t.r_stocks.include? rs }

  end
  
  def self.get_from_selection
    #Get object for train of selected rolling stock
    #Only one entity should be selected
    #Returns nil if not a rolling stock

    rs = RStock.get_from_selection
    return unless rs
    
    rs.train

  end

  def self.unique_name(name, model = nil)
    #Adds a number to name or increases the number if there already is one.
    #Used when train is split.
    #Assume active model
    model = Sketchup.active_model unless model

    name = S.tr("Untitled") if name == ""

    return name if @@instances.select { |i| i.model == model && i.name == name }.empty?

    number = 0
    matches = /#\d+\z/.match(name)
    number = matches[0].to_i if matches

    basename = name.sub /\s#\d+\z/, ""

    while true
      number += 1
      new_name =  "#{basename} ##{number}"
      break if @@instances.select { |i| i.model == model && i.name == new_name }.empty?
    end

    new_name

  end

  #Instance attribute accessors

  attr_accessor :r_stocks
  attr_reader :name#Has custom writer that also changes the rolling stocks' train name reference and make sure name is unique for model
  attr_reader :v#Speed (|velocity|) in m/s. Has custom writer to set v_target to the same value
  attr_accessor :v_target#Value v is accelerating towards by +-a
  attr_reader :a#|acceleration| in m/s^2(positive or negative doesn't matter). Has custom writer that makes value absolute
  attr_accessor :autopilot_mode
  attr_accessor :autopilot_route

  #Instance methods

  def initialize(r_stocks)
    #Creating train object from an array of connected rolling stocks

    @r_stocks = r_stocks
    rs = @r_stocks[0]
    @name = rs.train_name
    @v = rs.train_v || 0#This is speed and not actual velocity, but v looks much better in all formulas
    @v_target = rs.train_v_target || @v
    @a = rs.train_a || 0.25
    @autopilot_mode = rs.train_autopilot_mode#nil is no autopilot, "random" changes switches randomly and "guided" changes switches depending on data in route hash.
    @autopilot_mode = nil if @autopilot_mode == "guided"#Guided can't be saved as autopilot mode when model closes since references to track objects /the switches) cannot be saved and therefore not the route hash.
    @autopilot_route = nil

    @@instances << self

  end

  def a=(value)
    #Acceleration value is always positive. difference between v and v_target is used to determine whether v is increasing or decreasing
    @a = value.abs
  end
  
  def delete!
    #Remove train object from instances array and reference in drive train tool.
    #Called from RStock.delete! when the last rolling stock in a train is removed.
    #This method does NOT remove the rolling stocks associated with the train.
    
    @@instances.delete self
      
    #Remove train from followed by camera in drive tool
    ToolTrainDrive.active_train_for_models.delete_if { |i| i[:train] == self }

  end
  
  def distance_buffers
    #Returns first buffer distance of first rolling stock and last buffer distance of last rolling stock
    #Used to calculate train collisions

    [@r_stocks[0].distance_buffers[0], @r_stocks[-1].distance_buffers[1]]

  end

  def draw_lights
    #Show the front (white) light in front end of first rolling stock,
    #the end light (red) in back end of last rolling stock
    #and hides all other lamps.
    
    #Called from Train.reverse unless this method has an argument telling it not to
    #which is the case in Train.join where Train.reverse is called multiple times.
    #draw_lights is then called separately.
    
    #A light is a group or component instance in the model root or a tilting group (group named "tilting").
    #Lamps are named "lamp_<end of rs>_<front|back>,
    #<rs end> is the end of the rolling stock,
    #0 being closest to the rolling stock group origin
    #and 1 is further away.
    #front means this group is the light that should be shown when this end
    #of this rs is in the front of the train. back means this light should be shown when
    #in the back end of train.
    
    model.start_operation "Update Train Lights", true#Ideally a invisible operator that doesn't add to the undo queue for consistency with .move! in Rrain.go.
    
    #Find lamps in rolling stocks
    @r_stocks.each do |rs|
      next if rs.group.deleted?#is the case when splitting train because rolling stock was deleted
      
      tilting_body = rs.group.entities.find { |e| [Sketchup::Group, Sketchup::ComponentInstance].include?(e.class) && e.name == "tilting" }
      ents = rs.group.entities.to_a
      ents += tilting_body.entities.to_a if tilting_body
      
      ents.each do |e|
        next unless [Sketchup::Group,Sketchup::ComponentInstance].include? e.class
        next if e.name == ""
        matches = e.name.match(/light(0|1)_(front|back)/)
        next unless matches
               
        rs_end = matches[1].to_i
        rs_end = 1-rs_end if rs.reversed#Compensate rolling stock end if rolling stock is reversed (is going backwards), rs_end should be relative to the direction of travel
        light_end = matches[2] == "front" ? 0 : 1
        
        if light_end == rs_end && rs == @r_stocks[-rs_end]
          #is either front light in front end of rs in first rs of train
          #or back light in back end of rs in last rs of train
          e.hidden = false
        else
          e.hidden = true
        end
        
      end#each
    end#each
    
    model.commit_operation
    
    true
    
  end
  
  def draw_text(texts, wrap_in_operation = true)
    #Rewrites 3d text in all train's rolling stock group.
    #text is an array containing hashes of what to draw.
    #Each hash contains the text itself and a label.
    
    #Example: train.draw_text([{:label => "Number", :text => "007"}])
    
    Sketchup.active_model.start_operation(S.tr("Change Train Text"), true) if wrap_in_operation
    
    @r_stocks.each { |rs| rs.draw_text(texts, false) }#Don't wrap each rolling stock drawing in its own operation
    
    Sketchup.active_model.commit_operation if wrap_in_operation
    
  end
  
  def frame(delta_t)
    #Calculate velocity from acceleration if needed and then move train
    #Called from Animate class each frame
  
    #Update speed from acceleration (linear)
    unless @v == @v_target
      #acceleration value is always positive so it doesn't have to change when changing between increasing and decreasing speed.
      a = @a#No need to dup primitive classes
      if @v_target > @v
        #increasing speed
        @v += a*delta_t
        @v = @v_target if @v_target < @v
      else
        #Decreasing speed
        @v -= a*delta_t
        @v = @v_target if @v_target > @v
      end
    end
    
    #Move train
    distance = delta_t*(@v).m#Convert from m to inches here
    self.move distance
  
  end

  def go_to(point, preview = false)#NOTE: UNDER CONSTRUCTION:
    #Sets autopilot to guided for train.
    #Switches will be changed as the train reaches them to reach certain point.
    
    #Point is the point the front of the train will go to.
    #Preview set to true prevents the train from changing properties and is used to only preview a path.
    
    #Returns path if there is one, otherwise false.
    
    rs_points = @r_stocks[0].points
    route = Track.path_between_points rs_points[0], point, rs_points[0]-rs_points[1]
    
    return false unless route
    
    unless preview
      @autopilot_mode = "guided"
      train_route = {:switch_states => route[:switch_states]}#May contain more variables in the future such as point to decelerate from. If decelerate point is used. calculate distance to stop with current v and a. reverse path and limit it to that length to get point.
      @autopilot_route = train_route
    end
    
    #Return path
    route[:path]
    
  end
  
  def inspect
    #Get inspect string consistent to Sketchup's own classes with train name in
    "#<#{self.class}:0x#{(self.object_id << 1).to_s(16)} (#{@name})>"
  end

  def join(train, end_of_self, end_of_train)
    #Merge to train into one
    #Called from coupling tool and insert rolling stock tool
    #end_of_self is the end being connected of the train methods preformed on
    #end_of_train is the end of the other train
    #0 is front and 1 back

    #Reverse self so other train is added to its end
    self.reverse(false) if end_of_self == 0

    #Reverse train so it starts in the end being connected
    train.reverse(false) if end_of_train == 1

    #Add r_stock array from train to self's
    @r_stocks += train.r_stocks

    #Update train_positions for rolling stocks added rolling stocks
    #This value is used when re-initializing train when model is reopened
    counter = 0
    @r_stocks.each do |rs|
      rs.train_position = counter
      counter += 1
    end

    #Compare names
    #If same base name (name without number), use lowest number
    #Otherwise name of longest train
    if @name.sub(/\s#\d+\z/, "") == train.name.sub(/\s#\d+\z/, "")
      #Same basename, use lowest number
      number_self = 0
      matches = /#\d+\z/.match(@name)
      number_self = matches[0].to_i if matches

      number_train = 0
      matches = /#\d+\z/.match(train.name)
      number_train = matches[0].to_i if matches

      self.name = train.name if number_train < number_self#self.name= also saves name to rolling stocks ######(nope, it doesn't seem to be saved, perhaps because the train already has the name?)
    else
      #Different basename, use the one of the longest train
      self.name = train.name if train.r_stocks.length > @r_stocks.length
    end

    #Save train name to rolling stocks. It appears as if this isn't done by self.name= in this case.
    @r_stocks.each do |rs|
      rs.train_name = @name
    end

    #Change active train in drive tool if it was the train now being removed
    unless ToolTrainDrive.active_train_for_models.empty?
      if ToolTrainDrive.active_train_for_models.find { |i| i[:model] == model } [:train] == train
        ToolTrainDrive.active_train_for_models.find { |i| i[:model] == model }[:train] = self
      end
    end

    #Reverse back (base direction on self, 'the train another train is added too')
    self.reverse(false) if end_of_self == 0

    #Remove train object
    Train.instances.delete train
    
    #Update lights
    self.draw_lights

    self

  end

  def list_texts
    #List texts and labels in train.
    #Return array.
    #Element looks like {:label => "Destination", :text => "Knuffingen"}.
    #If different rolling stocks has different texts for the same label, use nil as text.
    
    #List all texts
    texts = @r_stocks.map { |rs| rs.list_texts } .flatten
    
    #Group by label    
    #if texts differs for one label, use nil as text value
    i = 0
    stop = texts.length
    while i < stop
      #Loop all texts
      j = i + 1
      while j < stop
        #Loop all texts after element in outer loop
        if texts[i][:label].downcase == texts[j][:label].downcase
          #Both texts has the same label, set text to nil if text value differs. remove the last of the 2 matching elements.
          texts[i][:text] = nil unless texts[i][:text] == texts[j][:text]
          texts.delete_at j
          j -= 1
          stop -= 1
        end
        j += 1
      end
      i += 1
    end
  
    texts
  
  end
  
  def model
  
    @r_stocks[0].model
  
  end
  
  def move(distance)
    #Move whole train along track.
    #Distance in inches (SU uses inches internally)
    #Distance is the distance first point moves (first rolling stock's first buffer distance ahead of first rotation point),
    #the other rolling socks' distances are calculated from this so their linear distances between remains the same.
    #Train always moves a positive distance, to move the other way the r_stocks array is reversed along with the controls of each rolling stock.
    #In this way the distance the first rolling stock moves is always known and it can be stopped at track stops for instance.

    #Can only move positive distance. To go backwards, reverse train.
    return false if distance <= 0#NOTE: draw at standstill to reset tilting couches?
    
    #Train must be completely on track
    return false unless self.on_track?

    rs_first = @r_stocks[0]
    vector = rs_first.points[0] - rs_first.points[1]
    #First tangent vector could be used (instead of vector between rotation points) but code still wouldn't work for corners that sharp that back bogie somewhere needs to go backwards.

    #Check if train collides with other train
    Train.instances.each do |train|
      [0, -1].each do |train_end|
        #Can't collide with self
        next if train == self
        next unless train.model == model
        
        #Is distance until collision less than distance to move in this frame?
        distance_between_points = rs_first.points[0].distance(train.points[train_end])
        distance_between_buffers = distance_between_points - (rs_first.distance_buffers[0] + train.distance_buffers[train_end])
        next if distance_between_buffers > distance
        
        #Is other train on the same track?
        query = Track.calc_point_along(rs_first.points[0], distance_between_points, vector, nil, rs_first.tracks[0], model)
        next unless query[:point] == train.points[train_end]
        
        #If trains enter a trailing switch simultaneously they might end up next to each other with fronts "colliding" even though going in the same direction.
        #make one of them keep moving while one waits.
        if train_end == 0
          other_train_rs = train.r_stocks[0]
          other_train_vector = other_train_rs.points[1] - other_train_rs.points[0]
          angle = vector.angle_between other_train_vector
          angle
          if angle > 90.degrees
            if @name < train.name
              next
            else
              return false
            end
          end
        end
        
        #If already inside another train, wait.
        #NOTE: AESTETIC: code: how to see what train is "inside" what (which is the frontmost). hopefully this stops second pod from jumping.
        
        #If distance between is 0, don't move train, it has already collided.
        return false if distance_between_buffers.to_l == 0.to_l
        
        #Otherwise move until reaching other train
        distance = distance_between_buffers
        #if train_end == 0
        #  #Meeting
        #  @v = train.v = 0
        #else
        #  #Overtake
        #  @v = train.v
        #end
      end#each
    end#each    

    #Find point of first buffers (first control point + buffer distance along track)
    query = Track.calc_point_along(rs_first.points[0], rs_first.distance_buffers[0], vector, nil, rs_first.tracks[0], model)
    point_first_buffer_old = query[:point]

    #Move point of first along track
    #In theory arc length should be used instead of linear length but for these small distances (in each frame) it's hardly any difference.#NOTE: use arc length anyway?
    query1 = Track.calc_point_along(point_first_buffer_old, distance , vector, nil, query[:track], model)
    point_first_buffer_new = query1[:point]
    
    #Entering a new track?
    if query1[:track] != query[:track]

      track_end_left = query[:track].connections[0].include?(query1[:track]) ? 0 : 1
      track_end_entered = query1[:track].connections[0].include?(query[:track]) ? 0 : 1
      
      #Prevent trains from going into each other at trailing switch
      #Also seem to split trains that have already merged.
      #Wait if there's already a train spanning the end of the track that was about to be entered.
      return false unless query1[:track].trains_spanning_end(track_end_entered).empty?
      
      #If entering a facing switch with auto-pilot on, change switch accordingly.
      if @autopilot_mode && query[:track].connections[track_end_left].length > 1

        #Change switch
        state = case @autopilot_mode
                when "random"
                  rand query[:track].connections.length
                when "guided"
                  switch_data = @autopilot_route[:switch_states].find { |i| i[0] == query[:track] && i[1] == track_end_left }
                  switch_data[2] if switch_data
                end
        query[:track].set_switch(state, track_end_left) if state
        
        #Get new point_first_buffer_new since train now will travel another direction
        query1 = Track.calc_point_along(point_first_buffer_old, distance , vector, nil, query[:track], model)
        point_first_buffer_new = query1[:point]
    
      end
      
    end
        
    #Don't perform calculations if train didn't move (if it hit end of track)
    return false if point_first_buffer_new == point_first_buffer_old
    
    #NOTE: UNDER CONSTRUCTION: check balise here? check for decelerate point too. remove when passed.
    
    #Set start values for variables used in loop
    point_previous = point_first_buffer_new
    distance_buffer_previous = 0.m

    #Prepare rolling stock positions
    #If a point fails to move along track the train stops.
    #Preparing in a separate loop also helps pointing couples towards neighboring rolling stock.
    r_stocks_prepared = []
    @r_stocks.each do |rs|

      #Vector between rolling stock fix points, used to tell approximate direction of travel
      vector = rs.points[0] - rs.points[1]

      #Find new points
      #Stop if point couldn't be found (can happen if switch changes with train on it)
      points = []
      tracks = []
      tangents = []
      
      query = Track.calc_point_along(rs.points[0], distance_buffer_previous + rs.distance_buffers[0], vector, point_previous, rs.tracks[0], model)
      return false unless query
      points[0], tracks[0], tangents[0] = query[:point], query[:track], query[:tangent]
      
      query = Track.calc_point_along(rs.points[1], vector.length, vector, points[0], rs.tracks[1], model)
      return false unless query
      points[1], tracks[1], tangents[1] = query[:point], query[:track], query[:tangent].reverse
      
      #Replace hard tangents (those matching polyline path) with smooth (parametric)
      smooth_tangents = []
      0.upto(1) do |i|
        smooth_tangents[i] = tracks[i].smooth_tangent(points[i], tangents[i])
      end
      tangents = smooth_tangents
      
      #Save data for next iteration
      point_previous = points[1]
      distance_buffer_previous = rs.distance_buffers[1]

      #Save rolling stock data
      r_stocks_prepared << {:points => points, :tracks => tracks, :tangents => tangents}

    end#each

    #NOTE: UNDER CONSTRUCTION: check if last axis move over balise. check if moves over facing switch and then reset it to previous state if changed by auto pilot
    
    #Move rolling stocks
    (0..@r_stocks.length-1).each do |i|
      #Current rolling stock in loop
      rs = @r_stocks[i]
      #Hash with position to use for current rolling stock
      rs_prepared = r_stocks_prepared[i]

      #Find coupling orientations
      vectors_coupling = []
      #First end
      vectors_coupling[0] =
      if i == 0
        #First
        rs_prepared[:points][0] - rs_prepared[:points][1]
      else
        #Inside train
        rs_prev_prepared = r_stocks_prepared[i - 1]
        rs_prev_prepared[:points][1] - rs_prepared[:points][0]
      end
      #Second end
      vectors_coupling[1] =
      if i == @r_stocks.length-1
        #Last
        rs_prepared[:points][1] - rs_prepared[:points][0]
      else
        #Inside train
        rs_next_prepared = r_stocks_prepared[i + 1]
        rs_next_prepared[:points][0] - rs_prepared[:points][1]
      end
      
      #Keep track of distance traveled so moving parts such as wheels can be positioned correctly
      if rs.reversed
        rs.distance_traveled -= distance
      else
        rs.distance_traveled += distance
      end

      #Update position
      rs.points = rs_prepared[:points]
      rs.tracks = rs_prepared[:tracks]
      rs.tangents = rs_prepared[:tangents]
      rs.vectors_coupling = vectors_coupling

      #Move rolling stock to updated position
      rs.draw_to_position

    end#each
    
    #Check if train went over balise and if so, execute custom ruby code attached to it
    #The point of the frontmost buffer is used, not the first rotation point
    Balise.instances.each do |b|
      next unless MyGeom.point_between_points?(b.point, point_first_buffer_new, point_first_buffer_old)
      #Because of lag train can move long distances per frame and distance from old location to balise can be shorter than from old to new, even if balise is on another track.
      next unless self.tracks_all.include? b.track
      b.execute self
    end#each

    #Return true if train moved
    true

  end

  def name=(name)
    #Changes name of train but also saves the new name to all rolling stocks in train.
    #Train name in rolling stocks are used when initializing trains when model is loaded.

    #Make name unique for model
    name = Train.unique_name(name, model) unless name == @name

    @name = name

    @r_stocks.each do |rs|
      rs.train_name = name
    end

  end

  def on_track?
    #Check if all train's rolling stocks are on track
    
    !@r_stocks.map { |rs| rs.tracks }.flatten.include? nil
    
  end
 
  def points
    #Returns first point of first rolling stock and last point of last rolling stock
    #Used to calculate train collisions

    [@r_stocks[0].points[0], @r_stocks[-1].points[1]]

  end
  
  def points_all
    #Returns array of bogie points in train.

    @r_stocks.map { |rs| rs.points }.flatten

  end
 
  def reverse(draw_lights = true)
    #Reverses the rolling stock array and controls of all rolling stocks in it.
    #Update rolling stocks' train_position so these are kept when re-opening model.
    #Used to change moving direction of train so first rolling stock is also first in array.
    #Does not affect model visually.
    #Called from train drive tool and from when connecting trains.

    @r_stocks.reverse!
    @r_stocks.each do |rs|
      rs.train_position = (@r_stocks.length-1)-rs.train_position
      rs.reverse_controls
    end

    #Update lights if not specifically told not to.
    #Lights aren't updated when reverse is called from Train.join since reverse is then called multiple times.
    self.draw_lights if draw_lights
 
    true

  end

  def split(position, draw_lights = true)
    #Split train at the position:th coupling.
    #Called in coupling tool
    #Returns new train

    return if position < 0
    return if position > @r_stocks.length - 2

    #Split of rolling stocks
    r_stocks = @r_stocks.slice!((position+1)..-1)

    #Update train_position for split of rolling stocks
    r_stocks.each do |rs|
      rs.train_position-= position
      rs.train_position-= 1
    end#each

    #Initialize new train
    train = Train.new(r_stocks)
    train.name = Train.unique_name @name
    
    #Update lights if not specifically told not to do so.
    #Not done when called from RStock.delete since that is called from an observer which may cause a crash.
    if draw_lights
      self.draw_lights
      train.draw_lights
    end

    train

  end

  def text_UI
    #Open UI to Set text on signs on the whole train.
    #Notify user about texts that differs between rolling stock (e.g. Number) and don't redraw unless altered by user.
    
    #Text to show in inputbox when value differs between rolling stocks. it can still be replaced with a new string but when left as this it won't be drawn.
    text_differs = ""#Empty string is consistent to Entity Info. something like "<Differs - Don't alter>" could also be used
    
    texts = self.list_texts

    if texts.empty?
      UI.messagebox(S.tr("There are no texts in this train.") + "\n" + S.tr("'Plugins > Railroad > Documentation > Rolling Stock' describes how to add texts to rolling stocks."))
      return
    end#if
    
    #Ask for new text
    labels = texts.map { |t| t[:label] }
    defaults = texts.map { |t| t[:text] ? t[:text] : text_differs}
    input = UI.inputbox(labels, defaults, S.tr("Change Train Text"))
    return unless input
    input.each_with_index do |v, i|
      texts[i][:text] = v
    end
    
    #Remove elements with text <text_differs>
    texts.delete_if { |t| t[:text] == text_differs }
    
    #Call draw method on each rs
    self.draw_text texts
    
    true
    
  end
  
  def tracks
    #Returns first track of first rolling stock and last track of last rolling stock
    #Used to calculate train collisions

    [@r_stocks[0].tracks[0], @r_stocks[-1].tracks[1]]

  end
  
  def tracks_all(include_buffers = true)
    #Returns array of all tracks train stands on, including tracks buffers are located over.
    
    tracks = []
    
    #First buffer
    if include_buffers
      rs = @r_stocks[0]
      tracks << Track.calc_point_along(rs.points[0], rs.distance_buffers[0], rs.points[0]-rs.points[1], nil, rs.tracks[0], model)[:track]
    end

    #All axes/bogies
    tracks += @r_stocks.map { |rs| rs.tracks }.flatten

    #Last buffer
    if include_buffers
      rs = @r_stocks[-1]
      tracks << Track.calc_point_along(rs.points[1], rs.distance_buffers[1], rs.points[1]-rs.points[0], nil, rs.tracks[1], model)[:track]
    end
    
    tracks.uniq!
    tracks.compact!
    
    #Fill gaps of short tracks underneath train but with no axis on them
    tracks_gaps = []
    0.upto(tracks.length-2) do |i|
      t0 = tracks[i]
      t1 = tracks[i+1]
      next if t0.connections.flatten.include? t1
      tracks_gaps += (t0.connections.flatten & t1.connections.flatten)
    end

    tracks + tracks_gaps
    
    tracks

  end

  def v=(v)
    #Sets both v and v_target
    #Usually setting v is used when setting train velocity but setting v_target can be used to reach that speed with a custom acceleration instead of immediately.
    @v = @v_target = v
  end
  
end#class

end#module
