# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

class ToolTrainCoupling
  #This tool is used to join and split trains

  #Height to visually raise coupling points (for ascetics)
  @@coupling_height = 1.055.m

  #Sketchup tool definitions
  
  def initialize

    #Array of all couplings in model
    #Each coupling is a hash containing position, rolling stocks and status (connected or not)
    @couplings = []

    #Index of coupling closest to to mouse
    @coupling_selected_index = nil

    #Pick helper
    @ph = nil

    #Input point
    @ip = nil

    #Statusbar texts
    @status_text = S.tr "Click on rolling stock to toggle coupling."

    @cursor = UI.create_cursor(File.join(CURSOR_DIR, "train_coupling.png"), 2, 2)

  end

  def onSetCursor

    UI.set_cursor @cursor

  end

  def activate

    #Initialize pick helper to check if rolling stock is hovered
    @ph = Sketchup.active_model.active_view.pick_helper

    #Initialize input point to find closest coupling to mouse
    @ip = Sketchup::InputPoint.new

    #Defined in mouse move, used in coupling_from_mouse
    @x = @y = @view = nil

    #Check for coupling closest to mouse frequently. just using mouse move wouldn't update it when train moves
    @timer = UI.start_timer(0.5, true){ self.coupling_from_mouse }

    #Reset statusbar text
    Sketchup.set_status_text(@status_text, SB_PROMPT)

    Sketchup.active_model.active_view.invalidate

  end

  def deactivate(view)

    UI.stop_timer(@timer)
    view.invalidate

  end

  def draw(view)

    #Update couplings
    self.couplings_update

    #Closed couplings
    points_green = @couplings.select { |c| c[:status] }.map { |c| c[:position] }
    unless points_green.empty?
      view.draw_points points_green, 10, 2, Sketchup::Color.new(0, 255, 0)
      view.draw_points points_green, 10, 1, Sketchup::Color.new(0, 127, 0)
    end

    #Open couplings
    points_red = @couplings.select { |c| !c[:status] }.map { |c| c[:position] }
    unless points_red.empty?
      view.draw_points points_red, 10, 2, Sketchup::Color.new(255, 0, 0)
      view.draw_points points_red, 10, 1, Sketchup::Color.new(127, 0, 0)
    end

    #Hovered coupling
    if @coupling_selected_index && @couplings[@coupling_selected_index]
      c_h = @couplings[@coupling_selected_index]
      view.draw_points [c_h[:position]], 10, 2, Sketchup::Color.new(0, 0, 255)
      view.draw_points [c_h[:position]], 10, 1, Sketchup::Color.new(0, 0, 127)
    end

  end

  def onMouseMove(flags, x, y, view)
    #Check what coupling is closest to mouse

    #Saves mouse coords so coupling can be found
    @x = x
    @y = y
    @view = view

    #Find coupling closest to mouse
    #This is its own method since it's also called from a timer, taking moving trains into concern
    self.coupling_from_mouse

    #If animation is running there is no need to invalidate view again
    view.invalidate unless Animate.get_from_model(view.model).go

  end

  def onLButtonDown(flags, x, y, view)
    #Connect or disconnect hovered coupling

    return unless @coupling_selected_index

    coupling = @couplings[@coupling_selected_index]

    if coupling[:status]
      #Split

      train = coupling[:r_stocks][0].train
      index = train.r_stocks.index(coupling[:r_stocks][0])
      train.split index

    else
      #Join

      coupling[:trains][0].join coupling[:trains][1], coupling[:train_ends][0], coupling[:train_ends][1]

    end

    @coupling_selected_index = nil

  end

  def resume(view)

    #Reset status text after tool has been temporarily deactivated
    Sketchup.set_status_text(@status_text, SB_PROMPT)
    view.invalidate

  end

  #Own definitions
  #Not called from Sketchup itself

  def coupling_from_mouse
    #Sets @coupling_selected_index from mouse position
    #Called from mouse move and from a timer since it changes when train moves

    #If called by timer before mouse has been moved, stop
    return unless @x

    @ph.do_pick(@x, @y)
    picked = @ph.best_picked
    @ip.pick @view, @x, @y

    @coupling_selected_index = nil

    return unless picked

    rs_h = RStock.get_from_group(picked)
    if rs_h
      couplings = @couplings.select { |c| c[:r_stocks].include? rs_h}
      coupling_selected = couplings.sort_by { |c| c[:position].distance @ip.position }[0]
      @coupling_selected_index = @couplings.index(coupling_selected)
    end

  end

  def couplings_update
    #Sets an array of coupling hashes, each containing rolling stock references and status
    #Status is true for couplings inside trains and false for those between touching trains
    #Called from draw which is called every time trains move

    @couplings = []

    #Add connected couplings
    Train.instances.each do |train|
      next unless train.model == Sketchup.active_model
      0.upto(train.r_stocks.length-2) do |index|
        rs0 = train.r_stocks[index]
        rs1 = train.r_stocks[index + 1]
        #Find point where couples meet
        vector = rs1.points[0] - rs0.points[1]
        point =
        if vector.valid?
          vector.length = rs0.distance_buffers[1]
          rs0.points[1].offset vector
        else
          rs0.points[1].clone
        end
        #raise point for aesthetics
        point.offset! Geom::Vector3d.new(0, 0, @@coupling_height)
        @couplings << { :position => point, :status => true, :r_stocks => [rs0, rs1] }
      end#each
    end#each

    #Add disconnected couplings where train touches
    #Loop every train combination and see if trains "touches" at their ends
    #NOTE: optimization: change variables for this  when trains collide?
    #NOTE: can't train at different models collide in this code?
    0.upto(Train.instances.length-1) do |t0i|
      (t0i+1).upto(Train.instances.length-1) do |t1i|
        0.upto(1) do |end0|
          0.upto(1) do |end1|
            t0 = Train.instances[t0i]
            t1 = Train.instances[t1i]

            p0 = t0.points[end0]
            p1 = t1.points[end1]
            
            d0 = t0.distance_buffers[end0]
            d1 = t1.distance_buffers[end1]

            distance_between_points = p0.distance p1
            distance_between_buffers = distance_between_points - (d0 + d1)

            next unless distance_between_buffers.to_l == 0.to_l
            #Distance between couplings, if they point to each other, is 0.
            #However they might be on different tracks

            vector = p1 - p0
            query = Track.calc_point_along(p0, distance_between_points, vector, nil, t0.tracks[end0], t0.model)
            next unless query[:point] == p1
            #Trains does touch

            #Add to @couplings
            rs0 = t0.r_stocks[-end0]
            rs1 = t1.r_stocks[-end1]
            #Find point where couples meet
            point =
            if vector.valid?
              p0.offset vector, d0
            else
              p0.clone
            end
            #raise point for aesthetics
            point.offset! Geom::Vector3d.new(0, 0, @@coupling_height)
            @couplings << { :position => point, :status => false, :r_stocks => [rs0, rs1], :trains => [t0, t1], :train_ends => [end0, end1] }

          end#each
        end#each
      end#each
    end#each

  end

end#class

end#module
