# Eneroth Railroad System

# Copyright Julia Christina Eneroth, eneroth3@gmail.com

module EneRailroad

#Instead of requiring a certain version of Eneroth Upright Extruder an up to date version of the required methods is included in Eneroth Railroad System.
#This script may be newer than the one in the uprigth extruder script.

def self.flattenVector(vector,plane)
  #Flatten vector to plane
  #plane's point does not affect output, only its normal does

  pointStart = plane[0]
  normal = plane[1]
  pointVectorEnd = pointStart.offset vector#Point at end of vector
  lineToPlane = [pointVectorEnd, normal]#Line from previous point towards plane
  pointIntesection = Geom.intersect_line_plane lineToPlane, plane#Point where previous line intersects vector
  outputVector = pointIntesection.- pointStart#vector from startpoint to intersection
  return outputVector
end

def self.removeOrSoftenSmooth(edge)
  #Removes edge if bounded faces are co-planar, otherwise soften
  unless edge.deleted?
    if edge.faces.length == 2
      onPlane = true
      edge.faces[1].vertices.each do |i|
        #Use classify_point instead of normal vector to see if face is co-planar. More precise
        onPlane = false if (edge.faces[0].classify_point i.position) == Sketchup::Face::PointNotOnPlane
      end#each
      if onPlane
        edge.erase!
      else
        edge.soft = edge.smooth = true
      end#if
    end#if
  end#unless
end

def self.extrude(extrusion, path, vectorUpright=nil)
  #Actual extrude function.
  #Either called from userInput or external script.
  #Extrusion is Face face about to be extruded, path an Array of Points3ds to follow, vectorUpright is the vector to align extrusion to (by default upright)
  #If path start and ends with same point it's considered closed.
  #If not closed, the face is extruded relative to the nearest end of the path.
  #If closed, the face is extruded relative to its nearest point in the path.

  #Set vectorUpright to vertical if not set
  vectorUpright = Geom::Vector3d.new(0,0,1) unless vectorUpright

  #Validate input
  if vectorUpright.class != Geom::Vector3d
    UI.messagebox("No valid vector set.")
    return
  end#if
  if extrusion.class != Sketchup::Face
    UI.messagebox("No valid extrusion face set.")
  end#if

  #Check whether path is closed
  closed = (path[0] == path[-1])

  #Get drawing context of extrusion face(group, componentdefinition or model)
  drwingContext = extrusion.parent
  ents = drwingContext.entities

  #Get material of extrusion face
  mat = extrusion.material
  matBack = extrusion.back_material

  #Prepare path
  if closed
    #Closed path, make the point along path closest to the face the start point (not limited to corners)
    #Closest to face means closest to first vertex returned #NOTE: use center of gravity?

    #Declare vars for closest point, distance and point index where to add the new point to path array
    #Start with first point
    indexOfClosest = 0
    pClosest = path[0]
    dClosest = extrusion.vertices[0].position.distance path[0]

    #Loop all segments
    (0..(path.length-2)).each do |i|
      #Get point on segment that is closest to face's first vertex
      p1 = path[i]
      p2 = path[i+1]
      line = [p1,(p2.-p1)]
      p = extrusion.vertices[0].position.project_to_line line
      #Check if point is on actual segment
      if(p1.distance(p) > p1.distance(p2))
        #Point is beyond p2, use p2 instead
        p = p2
      elsif(p2.distance(p) > p2.distance(p1))
        #Point is beyond p1, use p1 instead
        p = p1
      end#if elsif

      #Check if this point is closer than the closest one seen so far
      d = extrusion.vertices[0].position.distance p
      next if d > dClosest

      indexOfClosest = i
      pClosest = p
      dClosest = d
    end#each

    #Add closest point to path array if not already there (= if not a corner)
    unless(path.include? pClosest)
      indexOfClosest+= 1
      path.insert(indexOfClosest, pClosest)
    end#unless

    #Temporary remove last point in path since it's the same as the first
    path.pop
    #Rotate array so start point is first
    (1..indexOfClosest).each do
      path << path[0]
      path.shift
    end#each
    #Copy first point to end of path so it again start and ends at the same place
    path << path[0]
  else
    #Open path, make path end closest to extrusion the start point
    #Closest to face means closest to first vertex returned #NOTE: use center of gravity?
    if ((extrusion.vertices[0].position.distance path[-1]) < (extrusion.vertices[0].position.distance path[0]))
      path.reverse!
    end#if
  end#if

  #Points around face to extrude
  startProfilePoints = []
  extrusion.outer_loop.vertices.each do |i|
    startProfilePoints << i.position
  end#each

  #Smooth points, indexes of points between curve segments
  smoothPoints = []
  (0..(extrusion.outer_loop.vertices.length-1)).each do |i|
    vertex = extrusion.outer_loop.vertices[i]
    smoothPoints << i if vertex.edges[0].curve != nil and vertex.edges[1].curve != nil
  end#each

  #Array of edges to remove or soften after extrusion is performed. Removing inside loop sometimes causes errors
  edges_to_remove_or_soften = []
  
  #Hidden edges, indices of hidden edges around extrusion profile.
  hidden_indices = extrusion.outer_loop.edges.map { |e| e.hidden? }
  
  #Array of geometry to hide
  geometry_to_hide = []

  #Points of previous profile, used when drawing edges between profiles
  previousProfilePoints = startProfilePoints

  #Edges of previous profile, will be removed if co-planar
  #Starts as empty array since edges of first profile shouldn't be soften
  previousProfileEdges = []

  #Find first vector of path, used for rotation
  if closed
    #Closed path, use flatten average of vectors along segment before and after point
    vector1 = flattenVector((path[1].- path[0]), [Geom::Point3d.new, vectorUpright]).normalize
    vector2 = flattenVector((path[0].- path[-2]), [Geom::Point3d.new, vectorUpright]).normalize
    vectorRotationStart = Geom::linear_combination 1, vector1, 1, vector2
  else
    #Not closed path, use flatten vector along first segment
    vectorRotationStart = flattenVector((path[1].- path[0]), [Geom::Point3d.new, vectorUpright]).normalize
  end#if elsif

  #Find angle in start point if closed oath, used for scaling
  if closed
    angeInStartPoint = flattenVector((path[0].- path[-2]), [Geom::Point3d.new, vectorUpright]).angle_between flattenVector((path[1].- path[0]), [Geom::Point3d.new, vectorUpright])
    scaleStart = 1/(Math.cos(angeInStartPoint/2))
  end

  #If closed path, remove extrusion face
  extrusion.erase! if closed

  #Unless closed path, remove extrusion face if all its edges bind other faces (consistency with push-pull & followme)
  unless closed
    removeStartFace = true
    extrusion.outer_loop.edges.each do |j|
      removeStartFace = false unless j.faces.length == 2
    end#each
    extrusion.erase! if removeStartFace
  end#unless

  #Loop path each point but first (it's profile is already drawn since it's the face being extruded)
  #Draw a profiled on each point aligned to both the path and the upright vector
  #Connect this profile with previous profile
  (1..(path.length-1)).each do |i|

    #Translate transformation
    vectorMove = path[i].- path[0]
    translation = Geom::Transformation.translation vectorMove

    #Rotate transformation
    if i == (path.length-1)
      #Last point
      if closed
        #Closed path
        vectorRotation = vectorRotationStart
      else
        #Not closed path
        vectorRotation = flattenVector((path[i].- path[i-1]), [Geom::Point3d.new, vectorUpright]).normalize
      end#if
    else
      #Not last nor first point
      vector1 = flattenVector((path[i].- path[i-1]), [Geom::Point3d.new, vectorUpright]).normalize
      vector2 = flattenVector((path[i+1].- path[i]), [Geom::Point3d.new, vectorUpright]).normalize
      vectorRotation = Geom::linear_combination 1, vector1, 1, vector2
    end#if
    #Find angle to rotate(positive angle returned)
    rotationAngle = vectorRotation.angle_between vectorRotationStart
    #Check if angle should be negative
    negAnglePointForward = Geom::Point3d.new.offset vectorRotationStart
    negAnglePointPos = negAnglePointForward.transform(Geom::Transformation.rotation(Geom::Point3d.new, vectorUpright, Math::PI/2))
    negAnglePointNeg = negAnglePointForward.transform(Geom::Transformation.rotation(Geom::Point3d.new, vectorUpright, -Math::PI/2))
    negAnglePointAlongPath = Geom::Point3d.new.offset vectorRotation
    rotationAngle*= -1 if (negAnglePointAlongPath.distance negAnglePointNeg) < (negAnglePointAlongPath.distance negAnglePointPos)
    #Create transformation
    rotation = Geom::Transformation.rotation path[i], vectorUpright, rotationAngle

    #Scale transformation
    #To keep the sides of the extrusion parallel (consistency with followme)
    #Done as several sub-transformations that are later merged to the actual scaling transformation
    #angle path turns in this point, projected to plane perpendicular to vectorUprigh, doesn't matter if positive or negative
    if i == (path.length-1)
      #Last point
      if closed
        #Closed path
        angeInPoint = angeInStartPoint
      else
        #Not closed path
        angeInPoint = 0
      end#if
    else
      #Not last nor first point
      angeInPoint = flattenVector((path[i].- path[i-1]), [Geom::Point3d.new, vectorUpright]).angle_between flattenVector((path[i+1].- path[i]), [Geom::Point3d.new, vectorUpright])
    end
    scale = 1/(Math.cos(angeInPoint/2))
    #Divide scale with scale of first profile. If first profile is in corner its width wont be the with on extrusion between corners
    scale/= scaleStart if closed
    #create scale transformation from origin in x axis
    scaling = Geom::Transformation.scaling Geom::Point3d.new, scale, 1, 1
    #Rotate scale transformation
    rScaling = Geom::Transformation.axes path[i], (vectorRotation.cross vectorUpright), vectorRotation, vectorUpright
    scaling = rScaling.*scaling.*(rScaling.inverse)

    #Perform translation
    profilePoints = []
    startProfilePoints.each do |j|
      profilePoints << j.transform((scaling.*rotation).*(translation))
    end

    #Draw profile
    #NOTE: why draw faces?
    face = ents.add_face profilePoints
    pofilesEdges = face.edges
    face.erase!

    #Connect corresponding points to previous profile
    smoothExtrudeLines = []
    (0..(profilePoints.length-1)).each do |j|
      edgeStraight = ents.add_line profilePoints[j], previousProfilePoints[j]
      smoothExtrudeLines << edgeStraight if smoothPoints.include? j
    end#each

    #Draw diagonals and form faces towards previous profile
    diagonals = []
    (0..(profilePoints.length-1)).each do |j|
      #NOTE: better name for nextIndex
      nextIndex = j+1
      nextIndex = 0 if nextIndex == profilePoints.length
      edge = ents.add_line profilePoints[j], previousProfilePoints[nextIndex]
      diagonals << edge
      #Draw each face at a time instead of using edge.find_faces. The face connected to previous segment should be drawn first to prevent back face problems.
      ents.add_face profilePoints[j], previousProfilePoints[j], previousProfilePoints[nextIndex]
      ents.add_face profilePoints[j], profilePoints[nextIndex], previousProfilePoints[nextIndex]
      #Paint new faces
      edge.faces.each do |k|
        k.material = mat
        k.back_material = matBack
      end#each
      #Hide faces if they are extruded from a hidden edge
      if hidden_indices[j]
        geometry_to_hide += edge.faces
        geometry_to_hide += edge.faces.map { |f| f.edges }.flatten
      end
      
    end#each

    #Soften and remove co-planar edges in diagonals, previous profile, and edges extruded from curve
    #NOTE:
    # Add a separate array of what points in path should have its profile soften, soften if one of the edges connected to the point is a part of a curve (like native follow me)?
    # extrude() could be called with second parameter for softening profiles. All hard, all soft or array of soft corners. userInput() could set this depending on if vertices holds edges that are part of curves. modifier keys for all/none soften.
    edges_to_remove_or_soften+= diagonals
    edges_to_remove_or_soften+= previousProfileEdges
    edges_to_remove_or_soften+= smoothExtrudeLines

    #Save data for next iteration
    previousProfilePoints = profilePoints
    previousProfileEdges = pofilesEdges
  end#each

  #If closed path, soften first/last profile's edges
  if closed
    edges_to_remove_or_soften+= previousProfileEdges
  end#if

  #Remove or soften edges that shouldn't be visible
  edges_to_remove_or_soften.uniq!
  edges_to_remove_or_soften.each { |i| removeOrSoftenSmooth(i) }
  
  #Hide geometry extruded from hidden edge
  geometry_to_hide.uniq!
  geometry_to_hide.each do |i|
    next if i.deleted?
    #next if i.class == Sketchup::Edge && i.soft?#This made profiles visible as 1px points.
    i.hidden = true
  end

  #Add end face unless all its edges bind an outer face or path is closed(consistency with push-pull & followme)
  unless closed
    drawEndFace = false
    previousProfileEdges.each do |i|
      drawEndFace = true if i.faces.length != 2
    end#each
    if drawEndFace
      endFace = ents.add_face previousProfileEdges
      endFace.material = mat
      endFace.back_material = matBack
    end#if
  end#unless

  #Return end face. Can be used by other plugins calling the extruder
  return endFace if endFace

end

end#module
